<?php
require_once 'includes/config.php'; // Database connection and session start
require_once 'includes/auth.php';   // Authentication functions (like is_logged_in)

// Ensure the user is logged in
if (!is_logged_in()) {
    // Optionally set an error message if needed
    // $_SESSION['error'] = "You must be logged in to comment.";
    header("Location: login.php"); // Redirect to login if not logged in
    exit();
}

// Log start and session data for debugging
error_log("--- activity_add_comment.php started ---");
error_log("Current SESSION data: " . print_r($_SESSION, true));

// Check if the form was submitted via POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    error_log("Processing POST request...");
    error_log("Received POST data: " . print_r($_POST, true));

    // --- Data Retrieval and Validation ---
    $comment_text = trim($_POST['comment'] ?? '');
    $source_type = trim($_POST['source_type'] ?? '');
    $source_id = filter_input(INPUT_POST, 'source_id', FILTER_VALIDATE_INT);
    $comment_type = trim($_POST['comment_type'] ?? '');
    $parent_id = filter_input(INPUT_POST, 'parent_id', FILTER_VALIDATE_INT); // Get parent_id, validate as int
    $created_by = $_SESSION['username']; // Get username from session

    // Basic validation
    if (empty($comment_text) || empty($source_type) || $source_id === false || empty($comment_type) || empty($created_by)) {
        $_SESSION['error'] = "Gagal menambahkan komentar. Data tidak lengkap.";
        error_log("Validation failed - missing required fields. Comment: '$comment_text', SourceType: '$source_type', SourceID: '$source_id', CommentType: '$comment_type', CreatedBy: '$created_by'");
        // Redirect back even on validation failure
        if ($source_id !== false && !empty($source_type)) {
             // Try to redirect back to the correct detail page if possible
             $redirect_url = $source_type . "_detail.php?id=" . $source_id;
             // Basic check if the file might exist (improve as needed)
             if (file_exists($redirect_url)) {
                 header("Location: " . $redirect_url);
                 exit();
             }
        }
        // Fallback redirect
        header("Location: index.php"); // Or a more appropriate default page
        exit();
    }

    // If parent_id is not a valid integer (e.g., empty string from form), set it to NULL
    if ($parent_id === false || $parent_id <= 0) {
        $parent_id = null;
    }

    // --- Database Insertion ---
    try {
        $sql = "INSERT INTO unified_comments (source_type, source_id, parent_id, comment, comment_type, created_by, created_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW())";
        $stmt = $conn->prepare($sql);

        if (!$stmt) {
            throw new Exception("Prepare failed: (" . $conn->errno . ") " . $conn->error);
        }

        // Bind parameters: s = string, i = integer
        // Note: parent_id can be integer or null, bind accordingly
        $stmt->bind_param("siisss",
            $source_type,
            $source_id,
            $parent_id, // Bind the potentially NULL value
            $comment_text,
            $comment_type,
            $created_by
        );

        if ($stmt->execute()) {
            $_SESSION['success'] = "Komentar berhasil ditambahkan.";
            error_log("Comment inserted successfully for source $source_type/$source_id by $created_by. Rows affected: " . $stmt->affected_rows);
        } else {
            throw new Exception("Execute failed: (" . $stmt->errno . ") " . $stmt->error);
        }
        $stmt->close();

    } catch (Exception $e) {
        $_SESSION['error'] = "Terjadi kesalahan saat menyimpan komentar: " . $e->getMessage();
        // Log the detailed error
        error_log("Error inserting comment for source $source_type/$source_id by $created_by: " . $e->getMessage());
    }

    // --- Redirect Back ---
    // Construct the redirect URL based on source_type and source_id
    // Assuming detail pages follow the pattern like 'activity_detail.php', 'focus_detail.php' etc.
    $redirect_url = $source_type . "_detail.php?id=" . $source_id;

    // Basic check if the inferred file exists, otherwise redirect to a default page
    if (file_exists($redirect_url)) {
         error_log("Redirecting to: " . $redirect_url);
         header("Location: " . $redirect_url);
    } else {
         error_log("Redirect file not found: " . $redirect_url . ". Redirecting to index.php");
         $_SESSION['warning'] = "Komentar ditambahkan, tetapi halaman detail tidak ditemukan."; // Optional warning
         header("Location: index.php"); // Fallback redirect
    }
    exit();

} else {
    // If accessed directly via GET or other method
    error_log("activity_add_comment.php accessed via non-POST method.");
    $_SESSION['error'] = "Metode pengiriman tidak valid.";
    header("Location: index.php"); // Redirect to a safe default page
    exit();
}
?>