<?php
require_once 'includes/config.php';
// require_once 'includes/auth.php'; // Uncomment if login is required

// {{ Define the start and end year/month }}
$start_year = 2025;
$start_month = 7; // July
$end_year = 2026;
$end_month = 6; // June

$page_title = "Kalender Kegiatan Tahunan";
// {{ Set the date range strings for the query }}
$start_date = sprintf("%d-%02d-01", $start_year, $start_month);
$end_date_calc = new DateTime("$end_year-$end_month-01");
$end_date_calc->modify('last day of this month');
$end_date = $end_date_calc->format('Y-m-d');


// Fetch all activities for the defined range
$activities_by_date = [];
// {{ The SQL query uses the new $start_date and $end_date }}
$stmt_activities = $conn->prepare("SELECT id, name, date, status FROM activities WHERE date BETWEEN ? AND ? ORDER BY date ASC");
if ($stmt_activities) {
    $stmt_activities->bind_param("ss", $start_date, $end_date);
    $stmt_activities->execute();
    $result_activities = $stmt_activities->get_result();
    while ($activity = $result_activities->fetch_assoc()) {
        $activity_date = $activity['date'];
        if (!isset($activities_by_date[$activity_date])) {
            $activities_by_date[$activity_date] = [];
        }
        $activities_by_date[$activity_date][] = $activity;
    }
    $stmt_activities->close();
} else {
    // Handle error - query preparation failed
    error_log("Error preparing activities query for yearly calendar: " . $conn->error);
}

// {{ Fetch National Holidays within the range }}
$holidays_by_date = [];
$holiday_ranges_list = []; // {{ Store original holiday ranges here }}
$stmt_holidays = $conn->prepare(
    "SELECT start_date, end_date, description
     FROM national_holidays
     WHERE start_date <= ? AND end_date >= ? ORDER BY start_date ASC" // Added ORDER BY
);
if ($stmt_holidays) {
    $stmt_holidays->bind_param("ss", $end_date, $start_date);
    $stmt_holidays->execute();
    $result_holidays = $stmt_holidays->get_result();

    $view_start_dt = new DateTime($start_date);
    $view_end_dt = new DateTime($end_date);

    while ($holiday_range = $result_holidays->fetch_assoc()) {
        // {{ Store the original range }}
        $holiday_ranges_list[] = $holiday_range;

        // {{ Expand holiday ranges into individual dates for calendar/JS }}
        $holiday_start_dt = new DateTime($holiday_range['start_date']);
        $holiday_end_dt = new DateTime($holiday_range['end_date']);
        $interval = new DateInterval('P1D');
        $period_end_dt = clone $holiday_end_dt;
        $period_end_dt->modify('+1 day');
        $period = new DatePeriod($holiday_start_dt, $interval, $period_end_dt);

        foreach ($period as $date) {
            if ($date >= $view_start_dt && $date <= $view_end_dt) {
                 $date_str = $date->format('Y-m-d');
                 if (!isset($holidays_by_date[$date_str])) {
                     $holidays_by_date[$date_str] = $holiday_range['description'];
                 }
            }
        }
    }
    $stmt_holidays->close();
} else {
     error_log("Error preparing holidays query for yearly calendar: " . $conn->error);
}


// Function to generate a mini calendar for a given month and year
// {{ Add $holidays_by_date parameter }}
function generate_mini_calendar($month, $year, $activities_by_date, $holidays_by_date) {
    // No changes needed inside this function for month name map etc.
    $month_name_map = [
        1 => 'Januari', 2 => 'Februari', 3 => 'Maret', 4 => 'April', 5 => 'Mei', 6 => 'Juni',
        7 => 'Juli', 8 => 'Agustus', 9 => 'September', 10 => 'Oktober', 11 => 'November', 12 => 'Desember'
    ];
    $month_name = $month_name_map[$month] ?? date('F', mktime(0, 0, 0, $month, 1, $year));
    $first_day_of_month = date('N', mktime(0, 0, 0, $month, 1, $year)); // 1 (Mon) to 7 (Sun)
    $days_in_month = date('t', mktime(0, 0, 0, $month, 1, $year));

    $calendar = "<div class='mb-3 mini-calendar'>";
    $calendar .= "<h6>{$month_name} {$year}</h6>";
    $calendar .= "<table class='table table-bordered table-sm text-center'>";
    $calendar .= "<thead><tr><th>M</th><th>S</th><th>S</th><th>R</th><th>K</th><th>J</th><th>S</th></tr></thead>"; // Assuming Minggu start
    $calendar .= "<tbody><tr>";

    // Adjust for Sunday start (date('N') is 1-7 Mon-Sun, date('w') is 0-6 Sun-Sat)
    $first_day_index = date('w', mktime(0, 0, 0, $month, 1, $year)); // 0 (Sun) to 6 (Sat)

    // Add empty cells for the days before the first day of the month
    for ($i = 0; $i < $first_day_index; $i++) {
        $calendar .= "<td></td>";
    }

    $current_day = 1;
    $day_of_week = $first_day_index; // Start from the correct day index (0-6)

    while ($current_day <= $days_in_month) {
        // Start new row on Sunday
        if ($day_of_week == 7) {
            $day_of_week = 0;
            $calendar .= "</tr><tr>";
        }

        $date_str = sprintf("%d-%02d-%02d", $year, $month, $current_day);
        $cell_class = '';
        $clickable_attr = '';
        $title_attr = ''; // Initialize title attribute

        // {{ Add class for Sunday }}
        if ($day_of_week == 0) { // 0 corresponds to Sunday from date('w')
            $cell_class .= ' is-sunday';
        }

        // Check for holiday first
        if (isset($holidays_by_date[$date_str])) {
            // {{ Only add bold class for holiday, remove background/color class }}
            $cell_class .= ' is-holiday'; // Keep is-holiday for bold styling
            $title_attr = $holidays_by_date[$date_str]; // Set title to holiday description
        }

        // Check for activity
        if (isset($activities_by_date[$date_str])) {
            $activities = $activities_by_date[$date_str];
            $tooltip = '';
            $activity_id = '';
            
            if (!empty($activities)) {
                $activity_id = $activities[0]['id'];
                foreach ($activities as $activity) {
                    $tooltip .= $activity['name'] . "\n";
                }
                $tooltip = htmlspecialchars(trim($tooltip));
            }
            
            $cell_class .= ' has-activity';
            $clickable_attr = ' style="cursor:pointer;" onclick="window.location.href=\'activity_detail.php?id=' . $activity_id . '\'"';
            $title_attr = $tooltip;
        }

        // Add title attribute if set
        $title_html = $title_attr ? " title='" . htmlspecialchars($title_attr) . "'" : '';

        // Output the cell once
        $calendar .= "<td class='" . trim($cell_class) . "'{$clickable_attr}{$title_html}>{$current_day}</td>";

        $current_day++;
        $day_of_week++;
    }

    // Add empty cells for the remaining days of the week
    while ($day_of_week < 7 && $day_of_week != 0) { // Ensure we fill up to 7 cells (0-6)
        $calendar .= "<td></td>";
        $day_of_week++;
    }

    $calendar .= "</tr></tbody></table></div>";
    return $calendar;
}


// {{ Add PHP helper functions if not already included }}
if (!function_exists('formatDate')) {
    function formatDate($dateStr) {
        $date = new DateTime($dateStr);
        $months = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
        return $date->format('j') . ' ' . $months[$date->format('n') - 1] . ' ' . $date->format('Y');
    }
}

if (!function_exists('formatStatus')) {
    function formatStatus($status) {
        // Assuming status values like 'planned', 'ongoing', 'completed', 'cancelled'
        $map = [
            'planned' => 'Terencana',
            'ongoing' => 'Berlangsung',
            'completed' => 'Selesai',
            'cancelled' => 'Dibatalkan',
            // Add other statuses as needed
        ];
        return $map[$status] ?? ucfirst($status);
    }
}

if (!function_exists('getStatusColor')) {
    function getStatusColor($status) {
        // Map status to Bootstrap background color classes
        $map = [
            'planned' => 'primary',
            'ongoing' => 'success',
            'completed' => 'secondary',
            'cancelled' => 'danger',
            // Add other statuses as needed
        ];
        return $map[$status] ?? 'light';
    }
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($page_title) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    {{ This line includes your style.css file }}
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <style>
        .mini-calendar table { font-size: 0.8rem; }
        .mini-calendar th, .mini-calendar td { padding: 0.2rem !important; }
        .has-activity { font-weight: bold; } /* Keep bold for activity days */

        /* {{ Remove the holiday and Sunday styles from here }} */
        /* .is-holiday { ... } */
        /* .is-sunday { ... } */

        #activity-details-container { min-height: 400px; border-left: 1px solid #dee2e6; padding-left: 15px; }
        #activity-details-content { max-height: 80vh; overflow-y: auto; } /* Add scroll for details */
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    
                    <h1 class="h4"><?= htmlspecialchars($page_title) ?> - Juli <?= $start_year ?> s/d Juni <?= $end_year ?></h1>
                    
                </div>

                <div class="row">
                    <!-- Left Column: 12 Mini Calendars -->
                    <div class="col-lg-8">
                        <div class="row row-cols-1 row-cols-md-2 row-cols-xl-3">
                            <?php
                            // {{ Modify the loop to iterate from Jul 2025 to Jun 2026 }}
                            $current_month = $start_month;
                            $current_year_loop = $start_year;
                            for ($i = 0; $i < 12; $i++) {
                                echo "<div class='col'>";
                                // {{ Pass $holidays_by_date to the function }}
                                echo generate_mini_calendar($current_month, $current_year_loop, $activities_by_date, $holidays_by_date);
                                echo "</div>";

                                // Increment month and year
                                $current_month++;
                                if ($current_month > 12) {
                                    $current_month = 1;
                                    $current_year_loop++;
                                }
                            }
                            ?>
                        </div>
                    </div>

                    <!-- Right Column: Activity Details -->
                    <div class="col-lg-4">
                        <div id="activity-details-container">
                            <h5>Daftar Kegiatan & Libur (<?= formatDate($start_date) ?> - <?= formatDate($end_date) ?>)</h5>
                            <div id="activity-details-content">
                                
                                <?php
                                $display_items = [];

                                // Add holiday ranges to the display list
                                foreach ($holiday_ranges_list as $holiday) {
                                    $display_items[$holiday['start_date']][] = [
                                        'type' => 'holiday',
                                        'start_date' => $holiday['start_date'],
                                        'end_date' => $holiday['end_date'],
                                        'description' => $holiday['description']
                                    ];
                                }

                                // Add activity dates to the display list
                                foreach ($activities_by_date as $date => $activities) {
                                    if (!isset($display_items[$date])) {
                                        $display_items[$date] = []; // Ensure the date key exists
                                    }
                                    // Check if we already added a holiday starting on this date
                                    $has_holiday_entry = false;
                                    foreach($display_items[$date] as $item) {
                                        if ($item['type'] === 'holiday') {
                                            $has_holiday_entry = true;
                                            break;
                                        }
                                    }
                                    // Only add activity entry if no holiday entry exists for the same start date
                                    // (Activities will be shown under the holiday if they coincide)
                                    // Or, decide if you want separate entries regardless. Let's add it separately for now.
                                     $display_items[$date][] = [
                                         'type' => 'activity_date',
                                         'date' => $date,
                                         'activities' => $activities
                                     ];
                                }

                                // Sort the items by date (the array key)
                                ksort($display_items);

                                $found_any = false;
                                if (!empty($display_items)) {
                                    $found_any = true;
                                    echo '<div class="list-group list-group-flush">'; // Use list group for structure

                                    foreach ($display_items as $date => $items_on_date) {
                                        foreach ($items_on_date as $item) {
                                            if ($item['type'] === 'holiday') {
                                                echo '<div class="list-group-item p-1 alert alert-danger mb-1">'; // Holiday styling
                                                echo "<strong>Libur: ";
                                                if ($item['start_date'] == $item['end_date']) {
                                                    echo formatDate($item['start_date']); // Single day
                                                } else {
                                                    echo formatDate($item['start_date']) . " - " . formatDate($item['end_date']); // Date range
                                                }
                                                echo ":</strong> " . htmlspecialchars($item['description']);
                                                echo '</div>';
                                            } elseif ($item['type'] === 'activity_date') {
                                                // Check if this specific date is a holiday
                                                $is_holiday_today = isset($holidays_by_date[$item['date']]);

                                                // Display activities for this date
                                                foreach ($item['activities'] as $activity) {
                                                    $li_class = $is_holiday_today ? 'holiday-activity-item' : '';
                                                    echo "<div class='list-group-item p-1 " . $li_class . "'>";
                                                    echo "<strong>Kegiatan: " . formatDate($item['date']) . ":</strong> " . htmlspecialchars($activity['name']);
                                                    echo " <span class='badge bg-" . getStatusColor($activity['status']) . "'>" . htmlspecialchars(formatStatus($activity['status'])) . "</span>";
                                                    echo "</div>";
                                                }
                                            }
                                        }
                                    }
                                    echo '</div>'; // Close list-group
                                }

                                if (!$found_any) {
                                    echo '<p class="text-muted">Tidak ada kegiatan atau libur nasional dalam rentang waktu yang dipilih.</p>';
                                }
                                ?>
                                
                            </div>
                        </div>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const calendarCells = document.querySelectorAll('.mini-calendar td[data-date]');
            const detailsContainer = document.getElementById('activity-details-content');
            const activitiesData = <?= json_encode($activities_by_date, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE) ?>;
            // {{ Pass holiday data to JavaScript }}
            const holidaysData = <?= json_encode($holidays_by_date, JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE) ?>;

            calendarCells.forEach(cell => {
                cell.addEventListener('click', function() {
                    const date = this.getAttribute('data-date');
                    // {{ Pass holidaysData to the display function }}
                    displayActivityDetails(date, holidaysData);

                    // Optional: Highlight selected cell
                    calendarCells.forEach(c => c.classList.remove('bg-primary', 'text-white'));
                    // Avoid highlighting if it's just a holiday without activities? Or highlight anyway? Let's highlight.
                    this.classList.add('bg-primary', 'text-white');
                });
            });

            // {{ Modify display function to potentially show holiday info AND style activities }}
            function displayActivityDetails(date, holidays) {
                let html = `<h6>Detail pada ${formatDate(date)}:</h6>`;
                let contentFound = false;
                // {{ Check if the current date is a holiday }}
                const isHoliday = !!holidays[date]; // True if date exists in holidaysData

                // Check for holiday description first
                if (isHoliday) {
                    html += `<div class="alert alert-danger p-2 mb-2"><strong>Libur Nasional:</strong> ${escapeHtml(holidays[date])}</div>`;
                    contentFound = true;
                }

                // Check for activities
                if (activitiesData[date]) {
                    html += '<ul class="list-group list-group-flush">';
                    activitiesData[date].forEach(activity => {
                        // {{ Add 'holiday-activity-item' class if it's a holiday }}
                        const liClass = isHoliday ? 'holiday-activity-item' : '';
                        html += `<li class="list-group-item ${liClass}">`; // Add the class here
                        html += `<strong>${escapeHtml(activity.name)}</strong><br>`;
                        html += `<span class="badge bg-${getStatusColor(activity.status)}">${escapeHtml(formatStatus(activity.status))}</span><br>`;
                        // {{ Link remains the same }}
                        html += `<a href="activity_detail.php?id=${activity.id}" class="btn btn-sm btn-outline-info mt-1">Lihat Detail Lengkap</a>`;
                        html += `</li>`;
                    });
                    html += '</ul>';
                    contentFound = true;
                }

                if (!contentFound) {
                    // {{ Message remains the same }}
                    detailsContainer.innerHTML = `<p class="text-muted">Tidak ada kegiatan atau libur nasional pada tanggal ${formatDate(date)}.</p>`;
                } else {
                     detailsContainer.innerHTML = html;
                }
            }

            // {{ Other JS functions (formatDate, formatStatus, getStatusColor, escapeHtml) remain the same }}
            function formatDate(dateString) {
                const options = { year: 'numeric', month: 'long', day: 'numeric' };
                return new Date(dateString + 'T00:00:00').toLocaleDateString('id-ID', options);
            }

            function formatStatus(status) {
                if (!status) return 'Unknown';
                let formatted = status.charAt(0).toUpperCase() + status.slice(1).replace('_', ' ');
                return formatted;
            }

             function getStatusColor(status) {
                switch (status) {
                    case 'pending': return 'warning text-dark';
                    case 'onprogress': return 'info text-dark';
                    case 'completed': return 'success';
                    case 'canceled': return 'danger';
                    case 'must be evaluated': return 'primary';
                    default: return 'secondary';
                }
            }

            function escapeHtml(unsafe) {
                if (typeof unsafe !== 'string') {
                     return '';
                 }
                return unsafe
                         .replace(/&/g, "&amp;")
                         .replace(/</g, "&lt;")
                         .replace(/>/g, "&gt;")
                         .replace(/"/g, "&quot;")
                         .replace(/'/g, "&#039;");
             }
        });
    </script>
</body>
</html>