<?php
require_once '../config.php';

// Clear any previous output
if (ob_get_length()) ob_clean();

header('Content-Type: application/json');

try {
    // Verify request method
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception('Invalid request method', 405);
    }

    // Get input data (handles both JSON and form-data)
    $input = $_POST;
    if (empty($input)) {
        $input = json_decode(file_get_contents('php://input'), true) ?? [];
    }

    $id = filter_var($input['id'] ?? null, FILTER_VALIDATE_INT);
    $status = $input['status'] ?? '';
    
    if (!$id) {
        throw new Exception('Invalid task ID', 400);
    }

    // Validate status
    $validStatuses = ['pending', 'completed'];
    if (!in_array($status, $validStatuses)) {
        throw new Exception('Invalid status value', 400);
    }

    // Database update
    $stmt = $conn->prepare("UPDATE activity_todos SET status = ?, updated_at = NOW() WHERE id = ?");
    if (!$stmt) {
        throw new Exception('Database preparation failed', 500);
    }

    $stmt->bind_param("si", $status, $id);
    
    if (!$stmt->execute()) {
        throw new Exception('Failed to update status', 500);
    }

    // Successful response
    echo json_encode([
        'success' => true,
        'message' => 'Status updated successfully',
        'task_id' => $id,
        'new_status' => $status
    ]);
    
} catch (Exception $e) {
    // Error response
    http_response_code($e->getCode() ?: 500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'error_code' => 'STATUS_UPDATE_FAILED'
    ]);
} finally {
    if (isset($stmt)) $stmt->close();
    if (isset($conn)) $conn->close();
}