<?php
require_once 'includes/config.php';
if (!$dev_mode) {
    require_once 'includes/auth.php';
}

/**
 * Calculates an intermediate color along a gradient.
 *
 * @param string $startHex Hex code for the start color (e.g., '#add8e6').
 * @param string $endHex Hex code for the end color (e.g., '#00008b').
 * @param int $percentage Progress percentage (0-100).
 * @return string Hex code for the interpolated color.
 */
function getGradientColor($startHex, $endHex, $percentage) {
    $p = max(0, min(100, (int)$percentage)) / 100.0; // Normalize percentage to 0.0 - 1.0

    // Convert hex to RGB
    list($r1, $g1, $b1) = sscanf(ltrim($startHex, '#'), "%02x%02x%02x");
    list($r2, $g2, $b2) = sscanf(ltrim($endHex, '#'), "%02x%02x%02x");

    // Interpolate RGB components
    $r = round($r1 + ($r2 - $r1) * $p);
    $g = round($g1 + ($g2 - $g1) * $p);
    $b = round($b1 + ($b2 - $b1) * $p);

    // Convert back to hex
    return sprintf("#%02x%02x%02x", $r, $g, $b);
}


// Fetch activities with more details, todo success percentage, and item progress percentage
$activities_query = "
    SELECT 
        a.id, a.name, a.spotlight, a.date, a.person_in_charge, 
        COUNT(DISTINCT ut.id) as todo_count,
        SUM(CASE WHEN ut.status = 'completed' THEN 1 ELSE 0 END) as completed_count,
        COUNT(DISTINCT uil.id) as item_count,
        AVG(
            CASE uil.status
                WHEN 'pending' THEN 0
                WHEN 'ordered' THEN 50
                WHEN 'received' THEN 100
                WHEN 'cancelled' THEN 0
                ELSE 0
            END
        ) as item_progress_percentage,
        CASE 
            WHEN a.date < CURDATE() THEN 'past'
            ELSE 'current'
        END as activity_status
    FROM activities a
    LEFT JOIN unified_todos ut ON a.id = ut.source_id AND ut.source_type = 'activity'
    LEFT JOIN unified_item_list uil ON a.id = uil.source_id AND uil.source_type = 'activity'
    WHERE a.date >= DATE_SUB(CURDATE(), INTERVAL 6 MONTH)
    GROUP BY a.id, a.name, a.spotlight, a.date, a.person_in_charge
    ORDER BY a.date ASC
";
$activities = $conn->query($activities_query);

// Check for query errors (optional but recommended)
if (!$activities) {
    die("Error fetching activities: " . $conn->error);
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <style>
        .marquee-wrapper {
            overflow-x: auto;
            position: relative;
            padding: 0.25rem;
        }
        .marquee-content {
            white-space: nowrap;
        }
        .marquee-item {
            display: inline-block;
            margin-right: 2rem;
            padding: 0.25rem 0;
        }
        /* Hover effect for items */
        .marquee-item:hover {
            background-color: rgba(0, 0, 0, 0.05);
            border-radius: 4px;
            transition: background-color 0.2s;
        }
        /* Scrollbar styling */
        .marquee-wrapper::-webkit-scrollbar {
            height: 3px;
        }
        .marquee-wrapper::-webkit-scrollbar-track {
            background: #f1f1f1;
        }
        .marquee-wrapper::-webkit-scrollbar-thumb {
            background: #888;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4">Dashboard</h1>
                    <div class="btn-group">
                        <button class="btn btn-sm btn-outline-secondary active calendar-view" data-view="daily">Harian</button>
                        <button class="btn btn-sm btn-outline-secondary calendar-view" data-view="weekly">Mingguan</button>
                        <button class="btn btn-sm btn-outline-secondary calendar-view" data-view="monthly">Bulanan</button>
                        <button class="btn btn-sm btn-outline-secondary calendar-view" data-view="yearly">Tahunan</button>
                    </div>
                </div>

                <!-- Compact Calendar View -->
                <div class="card mb-4">
                    <div class="card-header py-2">
                        <div class="d-flex justify-content-between align-items-center">
                            <h5 class="mb-0">Kalender Deadline</h5>
                            <div class="btn-group btn-group-sm">
                                <button class="btn btn-outline-secondary active calendar-view" data-view="daily">H</button>
                                <button class="btn btn-outline-secondary calendar-view" data-view="weekly">M</button>
                                <button class="btn btn-outline-secondary calendar-view" data-view="monthly">B</button>
                                <button class="btn btn-outline-secondary calendar-view" data-view="yearly">T</button>
                            </div>
                        </div>
                    </div>
                    <div class="card-body p-2">
                        <div id="calendar" style="height: 300px;"></div>
                    </div>
                </div>

                <!-- Maintenance Marquee -->
                <div class="row g-2 mb-3"> <!-- Changed mb-4 to mb-3 and added g-2 for tighter gap -->
                    <!-- Facility Maintenance Column -->
                    <div class="col-md-6">
                        <div class="card">  <!-- Removed h-100 -->
                            <div class="card-header py-1"> <!-- Reduced padding -->
                                <h6 class="mb-0">Pemeliharaan Fasilitas</h6> <!-- Changed h5 to h6 -->
                            </div>
                            <div class="card-body py-1"> <!-- Added py-1 instead of p-0 -->
                                <div class="marquee-wrapper">
                                    <div class="marquee-content">
                                        <?php
                                        $maintenance = $conn->query("
                                            SELECT m.*, f.name as facility_name 
                                            FROM facility_maintenance m
                                            JOIN facilities f ON m.facility_id = f.id
                                            WHERE m.status != 'completed'
                                            ORDER BY m.maintenance_date ASC
                                            LIMIT 10
                                        ");
                                        while($item = $maintenance->fetch_assoc()): 
                                        ?>
                                        <span class="marquee-item">
                                            <span class="badge <?= $item['status'] == 'scheduled' ? 'bg-warning' : 'bg-danger' ?>">
                                                <?= ucfirst($item['status']) ?>
                                            </span>
                                            <strong><?= htmlspecialchars($item['facility_name']) ?></strong>
                                            (<?= date('d M', strtotime($item['maintenance_date'])) ?>)
                                            - <?= htmlspecialchars(substr($item['description'], 0, 30)) ?><?= strlen($item['description']) > 30 ? '...' : '' ?>
                                        </span>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Facility Construction Column -->
                    <div class="col-md-6">
                        <div class="card"> <!-- Removed h-100 -->
                            <div class="card-header py-1"> <!-- Reduced padding -->
                                <h6 class="mb-0">Jadwal Pembangunan Fasilitas</h6> <!-- Changed h5 to h6 -->
                            </div>
                            <div class="card-body py-1"> <!-- Added py-1 instead of p-0 -->
                                <div class="marquee-wrapper">
                                    <div class="marquee-content">
                                        <?php
                                        $construction = $conn->query("
                                            SELECT c.*, f.name as facility_name 
                                            FROM facility_construction c
                                            JOIN facilities f ON c.facility_id = f.id
                                            WHERE c.status != 'completed'
                                            ORDER BY c.start_date ASC
                                            LIMIT 10
                                        ");
                                        while($item = $construction->fetch_assoc()): 
                                        ?>
                                        <span class="marquee-item">
                                            <span class="badge <?= $item['status'] == 'planned' ? 'bg-info' : 'bg-primary' ?>">
                                                <?= ucfirst($item['status']) ?>
                                            </span>
                                            <strong><?= htmlspecialchars($item['facility_name']) ?></strong>
                                            (<?= date('d M', strtotime($item['start_date'])) ?>)
                                            - <?= htmlspecialchars(substr($item['description'], 0, 30)) ?><?= strlen($item['description']) > 30 ? '...' : '' ?>
                                        </span>
                                        <?php endwhile; ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Three Column Layout - Very Compact -->
                <div class="row">
                    <!-- Activities Column -->
                    <div class="col-md-4">
                        <h5 class="mb-3">Daftar Kegiatan</h5>
                        <ul class="list-group list-group-flush">
                            <?php
                            // Check if there are activities before attempting to loop
                            if ($activities && $activities->num_rows > 0) {
                                $activities->data_seek(0); // Reset pointer
                                while($activity = $activities->fetch_assoc()):
                                    // Calculate To-Do progress percentage
                                    $todo_progress_percentage = 0;
                                    if ($activity['todo_count'] > 0) {
                                        $todo_progress_percentage = round(($activity['completed_count'] / $activity['todo_count']) * 100);
                                    }
                                    
                                    // Calculate Item progress percentage (handle NULL if no items)
                                    $item_progress_percentage = 0;
                                    if ($activity['item_count'] > 0 && isset($activity['item_progress_percentage'])) {
                                         $item_progress_percentage = round((float)$activity['item_progress_percentage']);
                                    }

                                    // Define gradient colors
                                    $todoStartColor = '#add8e6'; // Light Blue
                                    $todoEndColor = '#00008b';   // Dark Blue
                                    $itemStartColor = '#90ee90'; // Light Green
                                    $itemEndColor = '#006400';   // Dark Green

                                    // Calculate current gradient end colors
                                    $currentTodoColor = getGradientColor($todoStartColor, $todoEndColor, $todo_progress_percentage);
                                    $currentItemColor = getGradientColor($itemStartColor, $itemEndColor, $item_progress_percentage);
                            ?>
                                <li class="list-group-item list-group-item-action py-2 px-0 <?= $activity['activity_status'] === 'past' ? 'bg-light' : '' ?>">
                                    <a href="activity_detail.php?id=<?= $activity['id'] ?>" class="text-decoration-none text-dark stretched-link">
                                        <div class="d-flex justify-content-between align-items-center mb-1">
                                            <span class="fw-bold"><?= htmlspecialchars($activity['name']) ?></span>
                                            <small class="text-muted"><?= date('d M', strtotime($activity['date'])) ?></small>
                                        </div>
                                        <?php if ($activity['activity_status'] === 'past'): ?>
                                            <span class="badge bg-secondary">Selesai</span>
                                        <?php endif; ?>
                                        
                                        <?php // Display To-Do progress bar ?>
                                        <div class="mb-1">
                                            <?php if ($activity['todo_count'] > 0): ?>
                                            <small class="text-muted d-block" style="font-size: 0.75em;">Tugas:</small>
                                            <div class="progress progress-sm" style="height: 5px;" title="Progress Tugas: <?= $todo_progress_percentage ?>%">
                                                <div class="progress-bar" role="progressbar" 
                                                     style="width: <?= $todo_progress_percentage ?>%; background: linear-gradient(to right, <?= $todoStartColor ?>, <?= $currentTodoColor ?>);" 
                                                     aria-valuenow="<?= $todo_progress_percentage ?>" aria-valuemin="0" aria-valuemax="100"></div>
                                            </div>
                                            <?php endif; ?>
                                        </div>

                                        <?php // Display Item progress bar ?>
                                        <div>
                                            <?php if ($activity['item_count'] > 0): ?>
                                            <small class="text-muted d-block" style="font-size: 0.75em;">Item:</small>
                                            <div class="progress progress-sm" style="height: 5px;" title="Progress Item: <?= $item_progress_percentage ?>%">
                                                <div class="progress-bar" role="progressbar" 
                                                     style="width: <?= $item_progress_percentage ?>%; background: linear-gradient(to right, <?= $itemStartColor ?>, <?= $currentItemColor ?>);" 
                                                     aria-valuenow="<?= $item_progress_percentage ?>" aria-valuemin="0" aria-valuemax="100"></div>
                                            </div>
                                            <?php endif; ?>
                                        </div>
                                    </a>
                                </li>
                            <?php
                                endwhile;
                            } else {
                                echo '<li class="list-group-item py-1 px-0 text-muted small">Tidak ada kegiatan.</li>';
                            }
                            ?>
                        </ul>
                    </div>

                    <!-- Procurement Column -->
                    <div class="col-md-4">
                         <h5 class="mb-3">Daftar Pengadaan</h5>
                         <ul class="list-group list-group-flush">
                            <?php
                            // Re-run query for procurements as it was used in calendar
                            $procurements_list = $conn->query("SELECT id, name, date FROM procurements ORDER BY date DESC LIMIT 5");
                            if ($procurements_list && $procurements_list->num_rows > 0) {
                                while($procurement = $procurements_list->fetch_assoc()):
                            ?>
                                <li class="list-group-item list-group-item-action py-1 px-0 d-flex justify-content-between align-items-center">
                                    <a href="procurement_detail.php?id=<?= $procurement['id'] ?>" class="text-decoration-none text-dark stretched-link">
                                        <?= htmlspecialchars($procurement['name']) ?>
                                    </a>
                                     <small class="text-muted"><?= date('d M', strtotime($procurement['date'])) ?></small>
                                </li>
                            <?php
                                endwhile;
                            } else {
                                 echo '<li class="list-group-item py-1 px-0 text-muted small">Tidak ada pengadaan.</li>';
                            }
                            ?>
                        </ul>
                    </div>

                    <!-- Focus Column -->
                    <div class="col-md-4">
                        <h5 class="mb-3">Daftar Fokus</h5>
                        <ul class="list-group list-group-flush">
                            <?php
                            $focuses = $conn->query("SELECT id, name, date FROM focuses ORDER BY date DESC LIMIT 5");
                            if ($focuses && $focuses->num_rows > 0) {
                                while($focus = $focuses->fetch_assoc()):
                            ?>
                                <li class="list-group-item list-group-item-action py-1 px-0 d-flex justify-content-between align-items-center">
                                    <a href="focus_detail.php?id=<?= $focus['id'] ?>" class="text-decoration-none text-dark stretched-link"> <!-- Assuming focus_detail.php exists -->
                                        <?= htmlspecialchars($focus['name']) ?>
                                    </a>
                                     <small class="text-muted"><?= date('d M', strtotime($focus['date'])) ?></small>
                                </li>
                            <?php
                                endwhile;
                            } else {
                                echo '<li class="list-group-item py-1 px-0 text-muted small">Tidak ada fokus.</li>';
                            }
                            ?>
                        </ul>
                    </div>
                </div>

                <!-- Categorized Activity Cards -->
                
            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.js"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const calendarEl = document.getElementById('calendar');
        const calendar = new FullCalendar.Calendar(calendarEl, {
            initialView: 'dayGridMonth',
            height: 'auto',
            headerToolbar: false,
            fixedWeekCount: false,
            dayMaxEvents: 2,
            eventDisplay: 'list-item',
            events: [
                <?php 
                // Activities events
                $activities->data_seek(0);
                while($activity = $activities->fetch_assoc()): 
                    if($activity['date']): 
                        // Set different colors for past and future events
                        $eventColor = strtotime($activity['date']) < time() ? '#6c757d' : '#3a87ad';
                ?>
                    {
                        title: '<?= addslashes($activity['name']) ?>',
                        start: '<?= $activity['date'] ?>',
                        color: '<?= $eventColor ?>',
                        extendedProps: {
                            type: 'activity'
                        }
                    },
                <?php 
                    endif;
                endwhile; 
                
                // Procurement events
                $procurements = $conn->query("SELECT * FROM procurements");
                while($procurement = $procurements->fetch_assoc()): 
                    if($procurement['due_date']): ?>
                    {
                        title: '<?= addslashes($procurement['name']) ?>',
                        start: '<?= $procurement['due_date'] ?>',
                        color: '#5cb85c',
                        extendedProps: {
                            type: 'procurement'
                        }
                    },
                    <?php endif;
                endwhile; ?>
            ],
            eventClick: function(info) {
                if(info.event.extendedProps.type === 'activity') {
                    window.location.href = 'activity_preparation.php?activity_id=' + info.event.id;
                } else {
                    window.location.href = 'procurement_detail.php?id=' + info.event.id;
                }
            }
        });
        calendar.render();
    
        // Handle view switching
        document.querySelectorAll('.calendar-view').forEach(btn => {
            btn.addEventListener('click', function() {
                const view = this.getAttribute('data-view');
                document.querySelectorAll('.calendar-view').forEach(b => b.classList.remove('active'));
                this.classList.add('active');
                
                switch(view) {
                    case 'daily':
                        calendar.changeView('dayGridDay');  // Changed from 'timeGridDay'
                        break;
                    case 'weekly':
                        calendar.changeView('dayGridWeek');  // Changed from 'timeGridWeek'
                        break;
                    case 'monthly':
                        calendar.changeView('dayGridMonth');
                        break;
                    case 'yearly':
                        calendar.changeView('listYear');
                        break;
                }
            });
        });
    });
    </script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>