<?php
require_once 'includes/config.php';


// Fetch current user data
try {
    // Remove 'email' from the SELECT list as the column doesn't exist
    $stmt = $conn->prepare("SELECT id, name FROM users WHERE username = ?");
    if (!$stmt) {
        throw new Exception("Prepare failed: (" . $conn->errno . ") " . $conn->error);
    }
    $stmt->bind_param("s", $username);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($result->num_rows === 1) {
        $user = $result->fetch_assoc();
    } else {
        throw new Exception("User not found."); // Should not happen if logged in
    }
    $stmt->close();
} catch (Exception $e) {
    $error = "Error fetching user data: " . $e->getMessage();
    // Log the detailed error
    error_log("Error fetching user data for $username: " . $e->getMessage());
    // Set $user to null explicitly on error to prevent later issues
    $user = null;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && $user) {
    $name = trim($_POST['name'] ?? '');
    // Remove email variable as it's not used
    // $email = trim($_POST['email'] ?? '');
    $current_password = $_POST['current_password'] ?? '';
    $new_password = $_POST['new_password'] ?? '';
    $confirm_password = $_POST['confirm_password'] ?? '';

    $update_fields = [];
    $params = [];
    $types = '';

    // --- Update Name ---
    if (!empty($name) && $name !== $user['name']) {
        $update_fields[] = "name = ?";
        $params[] = $name;
        $types .= 's';
    }

    // --- Remove Email Update Logic ---
    /*
    if (!empty($email) && $email !== $user['email']) {
        // Basic email validation
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $error = "Format email tidak valid.";
        } else {
             // Check if email already exists for another user
             $stmt_check = $conn->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
             $stmt_check->bind_param("si", $email, $user['id']);
             $stmt_check->execute();
             $result_check = $stmt_check->get_result();
             if ($result_check->num_rows > 0) {
                 $error = "Email sudah digunakan oleh pengguna lain.";
             } else {
                $update_fields[] = "email = ?";
                $params[] = $email;
                $types .= 's';
             }
             $stmt_check->close();
        }
    }
    */

    // --- Update Password ---
    if (!empty($current_password) && !empty($new_password) && !$error) {
        // Verify current password
        $stmt_pass = $conn->prepare("SELECT password FROM users WHERE id = ?");
        $stmt_pass->bind_param("i", $user['id']);
        $stmt_pass->execute();
        $result_pass = $stmt_pass->get_result();
        $current_user_data = $result_pass->fetch_assoc();
        $stmt_pass->close();

        if ($current_user_data && password_verify($current_password, $current_user_data['password'])) {
            if ($new_password === $confirm_password) {
                if (strlen($new_password) >= 6) { // Basic password length check
                    $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
                    $update_fields[] = "password = ?";
                    $params[] = $hashed_password;
                    $types .= 's';
                } else {
                    $error = "Password baru minimal harus 6 karakter.";
                }
            } else {
                $error = "Password baru dan konfirmasi password tidak cocok.";
            }
        } else {
            $error = "Password saat ini salah.";
        }
    } elseif (!empty($new_password) && empty($current_password)) {
         $error = "Masukkan password saat ini untuk mengubah password.";
    }


    // --- Execute Update ---
    if (!empty($update_fields) && !$error) {
        $sql = "UPDATE users SET " . implode(', ', $update_fields) . " WHERE id = ?";
        $types .= 'i';
        $params[] = $user['id'];

        try {
            $stmt_update = $conn->prepare($sql);
            if (!$stmt_update) {
                 throw new Exception("Prepare failed: (" . $conn->errno . ") " . $conn->error);
            }
            $stmt_update->bind_param($types, ...$params);

            if ($stmt_update->execute()) {
                $success = "Pengaturan berhasil diperbarui.";
                // Refresh user data after update
                // Remove 'email' from the refresh query
                $stmt_refresh = $conn->prepare("SELECT id, name FROM users WHERE id = ?");
                $stmt_refresh->bind_param("i", $user['id']);
                $stmt_refresh->execute();
                $result_refresh = $stmt_refresh->get_result();
                $user = $result_refresh->fetch_assoc(); // Update $user variable
                $stmt_refresh->close();

                // Update session name if changed (optional, depends on usage)
                // $_SESSION['user_name'] = $user['name'];

            } else {
                throw new Exception("Execute failed: (" . $stmt_update->errno . ") " . $stmt_update->error);
            }
            $stmt_update->close();
        } catch (Exception $e) {
            $error = "Gagal memperbarui pengaturan: " . $e->getMessage();
            // Log detailed error
            error_log("Error updating user settings for user ID {$user['id']}: " . $e->getMessage());
        }
    } elseif (empty($update_fields) && !$error && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $error = "Tidak ada perubahan yang dimasukkan.";
    }
}

// Include header
include 'includes/header.php';
?>

<div class="container-fluid">
    <div class="row">
        <?php include 'includes/sidebar.php'; ?>

        <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
            <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                <h1 class="h2">Pengaturan Pengguna</h1>
            </div>

            <?php if ($success): ?>
                <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>

            <?php if ($user): // Check if $user is not null before using it ?>
            <div class="card">
                <div class="card-body">
                    <form action="user_settings.php" method="POST">
                        <div class="mb-3">
                            <label for="username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="username" value="<?= htmlspecialchars($username) ?>" disabled readonly>
                            <small class="text-muted">Username tidak dapat diubah.</small>
                        </div>

                        <div class="mb-3">
                            <label for="name" class="form-label">Nama Lengkap</label>
                            <input type="text" class="form-control" id="name" name="name" value="<?= htmlspecialchars($user['name'] ?? '') ?>" required>
                        </div>

                        <!-- Remove Email Input Field -->
                        <!--
                        <div class="mb-3">
                            <label for="email" class="form-label">Email</label>
                            <input type="email" class="form-control" id="email" name="email" value="<?= htmlspecialchars($user['email'] ?? '') ?>" required>
                        </div>
                        -->

                        <hr>
                        <h5 class="mb-3">Ubah Password (Opsional)</h5>

                        <div class="mb-3">
                            <label for="current_password" class="form-label">Password Saat Ini</label>
                            <input type="password" class="form-control" id="current_password" name="current_password">
                            <small class="text-muted">Kosongkan jika tidak ingin mengubah password.</small>
                        </div>

                        <div class="mb-3">
                            <label for="new_password" class="form-label">Password Baru</label>
                            <input type="password" class="form-control" id="new_password" name="new_password">
                        </div>

                        <div class="mb-3">
                            <label for="confirm_password" class="form-label">Konfirmasi Password Baru</label>
                            <input type="password" class="form-control" id="confirm_password" name="confirm_password">
                        </div>

                        <button type="submit" class="btn btn-primary">Simpan Perubahan</button>
                    </form>
                </div>
            </div>
            <?php else: ?>
                <?php // Display a user-friendly message if user data couldn't be loaded ?>
                <div class="alert alert-warning">Gagal memuat data pengguna. Silakan coba lagi nanti atau hubungi administrator.</div>
                <?php if ($error): // Optionally display the specific error message again ?>
                    <!-- <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div> -->
                <?php endif; ?>
            <?php endif; ?>

        </main>
    </div>
</div>

<?php include 'includes/footer.php'; ?>