<?php
require_once 'includes/config.php';
// Uncomment the line below if you need authentication
// require_once 'includes/auth.php';

$page_title = "Tambah Kendaraan";
$error_message = '';
$success_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // --- Data Retrieval ---
    $name = trim($_POST['name'] ?? '');
    $license_plate = strtoupper(trim($_POST['license_plate'] ?? '')); // Often stored uppercase
    $brand = trim($_POST['brand'] ?? '');
    $year = !empty($_POST['year']) ? (int)$_POST['year'] : null;
    $stnk_number = trim($_POST['stnk_number'] ?? '');
    $engine_number = trim($_POST['engine_number'] ?? '');
    $chassis_number = trim($_POST['chassis_number'] ?? '');
    $status = $_POST['status'] ?? 'available'; // Default to available

    // --- Basic Validation ---
    if (empty($name) || empty($license_plate)) {
        $error_message = "Nama Kendaraan dan Nomor Polisi wajib diisi.";
    } elseif ($year !== null && ($year < 1900 || $year > date('Y') + 1)) { // Basic year check
        $error_message = "Tahun Kendaraan tidak valid.";
    } elseif (!in_array($status, ['available', 'maintenance', 'inactive'])) { // Only allow these statuses on add
        $error_message = "Status Kendaraan tidak valid.";
    } else {
        // --- Check for duplicate license plate ---
        $check_sql = "SELECT id FROM vehicles WHERE license_plate = ?";
        $stmt_check = $conn->prepare($check_sql);
        if ($stmt_check) {
            $stmt_check->bind_param("s", $license_plate);
            $stmt_check->execute();
            $stmt_check->store_result();
            if ($stmt_check->num_rows > 0) {
                $error_message = "Nomor Polisi '$license_plate' sudah terdaftar.";
            }
            $stmt_check->close();
        } else {
            $error_message = "Gagal memeriksa duplikasi Nomor Polisi: " . $conn->error;
        }

        // --- Proceed if no errors ---
        if (empty($error_message)) {
            $insert_sql = "INSERT INTO vehicles (name, license_plate, brand, year, stnk_number, engine_number, chassis_number, status) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, ?)";
            $stmt_insert = $conn->prepare($insert_sql);

            if ($stmt_insert) {
                // Bind parameters (s=string, i=integer) - year is treated as integer
                $stmt_insert->bind_param("sssissss", 
                    $name, 
                    $license_plate, 
                    $brand, 
                    $year, // Pass null if empty, otherwise integer
                    $stnk_number, 
                    $engine_number, 
                    $chassis_number, 
                    $status
                );

                if ($stmt_insert->execute()) {
                    $success_message = "Kendaraan '$name' berhasil ditambahkan.";
                    // Optional: Redirect after success
                    // header("Location: vehicles.php?status=added");
                    // exit;
                    
                    // Clear form fields for next entry (if not redirecting)
                    $_POST = []; // Clear POST data to reset form
                    $name = $license_plate = $brand = $year = $stnk_number = $engine_number = $chassis_number = '';
                    $status = 'available';

                } else {
                    $error_message = "Gagal menambahkan kendaraan: " . $stmt_insert->error;
                    error_log("Error adding vehicle: " . $stmt_insert->error); // Log detailed error
                }
                $stmt_insert->close();
            } else {
                $error_message = "Gagal mempersiapkan statement: " . $conn->error;
                error_log("Error preparing vehicle insert statement: " . $conn->error); // Log detailed error
            }
        }
    }
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4"><?= $page_title ?></h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="vehicles.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Kembali ke Daftar
                        </a>
                    </div>
                </div>

                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($error_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-body">
                        <form action="vehicle_add.php" method="POST">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="name" class="form-label">Nama Kendaraan <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="name" name="name" value="<?= htmlspecialchars($_POST['name'] ?? '') ?>" required>
                                    <div class="form-text">Contoh: Avanza Hitam, Innova Silver B 1234 ABC</div>
                                </div>
                                <div class="col-md-6">
                                    <label for="license_plate" class="form-label">Nomor Polisi <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="license_plate" name="license_plate" value="<?= htmlspecialchars($_POST['license_plate'] ?? '') ?>" required style="text-transform: uppercase;">
                                </div>
                                <div class="col-md-6">
                                    <label for="brand" class="form-label">Merk</label>
                                    <input type="text" class="form-control" id="brand" name="brand" value="<?= htmlspecialchars($_POST['brand'] ?? '') ?>">
                                </div>
                                <div class="col-md-6">
                                    <label for="year" class="form-label">Tahun</label>
                                    <input type="number" class="form-control" id="year" name="year" min="1900" max="<?= date('Y') + 1 ?>" step="1" value="<?= htmlspecialchars($_POST['year'] ?? '') ?>">
                                </div>
                                <div class="col-md-6">
                                    <label for="stnk_number" class="form-label">Nomor STNK</label>
                                    <input type="text" class="form-control" id="stnk_number" name="stnk_number" value="<?= htmlspecialchars($_POST['stnk_number'] ?? '') ?>">
                                </div>
                                <div class="col-md-6">
                                    <label for="engine_number" class="form-label">Nomor Mesin</label>
                                    <input type="text" class="form-control" id="engine_number" name="engine_number" value="<?= htmlspecialchars($_POST['engine_number'] ?? '') ?>">
                                </div>
                                <div class="col-md-6">
                                    <label for="chassis_number" class="form-label">Nomor Rangka</label>
                                    <input type="text" class="form-control" id="chassis_number" name="chassis_number" value="<?= htmlspecialchars($_POST['chassis_number'] ?? '') ?>">
                                </div>
                                <div class="col-md-6">
                                    <label for="status" class="form-label">Status Awal</label>
                                    <select class="form-select" id="status" name="status">
                                        <option value="available" <?= (($_POST['status'] ?? 'available') == 'available') ? 'selected' : '' ?>>Tersedia (Available)</option>
                                        <option value="maintenance" <?= (($_POST['status'] ?? '') == 'maintenance') ? 'selected' : '' ?>>Dalam Perawatan (Maintenance)</option>
                                        <option value="inactive" <?= (($_POST['status'] ?? '') == 'inactive') ? 'selected' : '' ?>>Tidak Aktif (Inactive)</option>
                                        <!-- 'on_loan' status is usually set via the loan system, not on initial add -->
                                    </select>
                                </div>
                            </div>

                            <hr class="my-4">

                            <button class="btn btn-primary" type="submit">
                                <i class="fas fa-save"></i> Simpan Kendaraan
                            </button>
                            <a href="vehicles.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Batal
                            </a>
                        </form>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>