<?php
require_once 'includes/config.php';
// Uncomment the line below if you need authentication
// require_once 'includes/auth.php';

$page_title = "Detail Kendaraan";
$error_message = '';
$vehicle_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);

$vehicle_data = null;
$last_loan = null;
$mileage_history = [];
$event_history = [];

if (!$vehicle_id) {
    $error_message = "ID Kendaraan tidak valid atau tidak ditemukan.";
    // Optional: Redirect if ID is missing
    // header("Location: vehicles.php?error=invalid_id");
    // exit;
} else {
    // 1. Fetch Basic Vehicle Data
    $stmt_vehicle = $conn->prepare("SELECT * FROM vehicles WHERE id = ?");
    if ($stmt_vehicle) {
        $stmt_vehicle->bind_param("i", $vehicle_id);
        $stmt_vehicle->execute();
        $result_vehicle = $stmt_vehicle->get_result();
        if ($result_vehicle->num_rows === 1) {
            $vehicle_data = $result_vehicle->fetch_assoc();
            $page_title = "Detail Kendaraan: " . htmlspecialchars($vehicle_data['name'] ?? 'N/A');
        } else {
            $error_message = "Kendaraan dengan ID $vehicle_id tidak ditemukan.";
        }
        $stmt_vehicle->close();
    } else {
        $error_message = "Gagal mempersiapkan query data kendaraan: " . $conn->error;
        error_log("Error preparing vehicle data statement: " . $conn->error);
    }

    // Proceed only if vehicle data was found
    if ($vehicle_data) {
        // 2. Fetch Last Loan Data
        $stmt_loan = $conn->prepare("
            SELECT borrower_name, start_date, end_date, purpose, status 
            FROM vehicle_loans 
            WHERE vehicle_id = ? 
            ORDER BY start_date DESC 
            LIMIT 1
        ");
        if ($stmt_loan) {
            $stmt_loan->bind_param("i", $vehicle_id);
            $stmt_loan->execute();
            $result_loan = $stmt_loan->get_result();
            if ($result_loan->num_rows === 1) {
                $last_loan = $result_loan->fetch_assoc();
            }
            $stmt_loan->close();
        } else {
             error_log("Error preparing last loan statement: " . $conn->error);
             // Non-fatal, just won't show loan info
        }

        // 3. Fetch Mileage History
        $stmt_mileage = $conn->prepare("
            SELECT reading_date, mileage, notes, created_at 
            FROM vehicle_mileage 
            WHERE vehicle_id = ? 
            ORDER BY reading_date DESC
        ");
         if ($stmt_mileage) {
            $stmt_mileage->bind_param("i", $vehicle_id);
            $stmt_mileage->execute();
            $result_mileage = $stmt_mileage->get_result();
            while ($row = $result_mileage->fetch_assoc()) {
                $mileage_history[] = $row;
            }
            $stmt_mileage->close();
        } else {
             error_log("Error preparing mileage history statement: " . $conn->error);
             // Non-fatal
        }

        // 4. Fetch Maintenance & Tax History
        $stmt_events = $conn->prepare("
            SELECT event_type, event_date, description, cost, next_due_date, notes, created_at 
            FROM vehicle_events 
            WHERE vehicle_id = ? 
            ORDER BY event_date DESC, created_at DESC
        ");
        if ($stmt_events) {
            $stmt_events->bind_param("i", $vehicle_id);
            $stmt_events->execute();
            $result_events = $stmt_events->get_result();
            while ($row = $result_events->fetch_assoc()) {
                $event_history[] = $row;
            }
            $stmt_events->close();
        } else {
             error_log("Error preparing event history statement: " . $conn->error);
             // Non-fatal
        }
    }
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        .detail-label { font-weight: bold; }
        .card-header i { margin-right: 5px; }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4"><?= $page_title ?></h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="vehicles.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Kembali ke Daftar Kendaraan
                        </a>
                        <?php if ($vehicle_data): ?>
                        <a href="vehicle_edit.php?id=<?= $vehicle_id ?>" class="btn btn-sm btn-outline-primary ms-2">
                            <i class="fas fa-pencil-alt"></i> Edit Kendaraan
                        </a>
                        <?php endif; ?>
                    </div>
                </div>

                <?php if ($error_message): ?>
                    <div class="alert alert-danger" role="alert">
                        <?= htmlspecialchars($error_message) ?>
                    </div>
                <?php endif; ?>

                <?php if ($vehicle_data): ?>
                <div class="row">
                    <!-- Vehicle Details Card -->
                    <div class="col-lg-6 mb-4">
                        <div class="card h-100">
                            <div class="card-header"><i class="fas fa-car"></i> Informasi Kendaraan</div>
                            <div class="card-body">
                                <dl class="row">
                                    <dt class="col-sm-4 detail-label">Nama/Model</dt>
                                    <dd class="col-sm-8"><?= htmlspecialchars($vehicle_data['name']) ?></dd>

                                    <dt class="col-sm-4 detail-label">No. Polisi</dt>
                                    <dd class="col-sm-8"><?= htmlspecialchars($vehicle_data['license_plate']) ?></dd>

                                    <dt class="col-sm-4 detail-label">Jenis</dt>
                                    
                                    <dd class="col-sm-8"><?= htmlspecialchars(ucfirst($vehicle_data['type'] ?? '-')) ?></dd>
                                    

                                    <dt class="col-sm-4 detail-label">Tahun</dt>
                                    <dd class="col-sm-8"><?= htmlspecialchars($vehicle_data['year']) ?></dd>

                                    <dt class="col-sm-4 detail-label">Status</dt>
                                    <dd class="col-sm-8">
                                        <?php 
                                            $status_badge = 'bg-secondary';
                                            if ($vehicle_data['status'] == 'available') $status_badge = 'bg-success';
                                            elseif ($vehicle_data['status'] == 'on_loan') $status_badge = 'bg-warning text-dark';
                                            elseif ($vehicle_data['status'] == 'maintenance') $status_badge = 'bg-info text-dark';
                                            elseif ($vehicle_data['status'] == 'inactive') $status_badge = 'bg-danger';
                                        ?>
                                        <span class="badge <?= $status_badge ?>"><?= htmlspecialchars(ucfirst(str_replace('_', ' ', $vehicle_data['status']))) ?></span>
                                    </dd>

                                    <dt class="col-sm-4 detail-label">Deskripsi</dt>
                                    <dd class="col-sm-8"><?= nl2br(htmlspecialchars($vehicle_data['description'] ?? '-')) ?></dd>
                                    
                                    <dt class="col-sm-4 detail-label">Ditambahkan</dt>
                                    <dd class="col-sm-8"><?= date('d M Y H:i', strtotime($vehicle_data['created_at'])) ?></dd>
                                </dl>
                            </div>
                        </div>
                    </div>

                    <!-- Last Loan Card -->
                    <div class="col-lg-6 mb-4">
                        <div class="card h-100">
                            <div class="card-header"><i class="fas fa-clipboard-list"></i> Peminjaman Terakhir</div>
                            <div class="card-body">
                                <?php if ($last_loan): ?>
                                    <dl class="row">
                                        <dt class="col-sm-4 detail-label">Peminjam</dt>
                                        <dd class="col-sm-8"><?= htmlspecialchars($last_loan['borrower_name']) ?></dd>

                                        <dt class="col-sm-4 detail-label">Tanggal Mulai</dt>
                                        <dd class="col-sm-8"><?= date('d M Y', strtotime($last_loan['start_date'])) ?></dd>

                                        <dt class="col-sm-4 detail-label">Tanggal Selesai</dt>
                                        <dd class="col-sm-8"><?= date('d M Y', strtotime($last_loan['end_date'])) ?></dd>

                                        <dt class="col-sm-4 detail-label">Keperluan</dt>
                                        <dd class="col-sm-8"><?= nl2br(htmlspecialchars($last_loan['purpose'] ?? '-')) ?></dd>
                                        
                                        <dt class="col-sm-4 detail-label">Status Pinjam</dt>
                                        <dd class="col-sm-8">
                                            <?php 
                                                $loan_status_badge = 'bg-secondary';
                                                if ($last_loan['status'] == 'scheduled') $loan_status_badge = 'bg-primary';
                                                elseif ($last_loan['status'] == 'ongoing') $loan_status_badge = 'bg-warning text-dark';
                                                elseif ($last_loan['status'] == 'completed') $loan_status_badge = 'bg-success';
                                                elseif ($last_loan['status'] == 'cancelled') $loan_status_badge = 'bg-danger';
                                            ?>
                                            <span class="badge <?= $loan_status_badge ?>"><?= htmlspecialchars(ucfirst($last_loan['status'])) ?></span>
                                        </dd>
                                    </dl>
                                <?php else: ?>
                                    <p class="text-muted text-center">Belum ada riwayat peminjaman untuk kendaraan ini.</p>
                                <?php endif; ?>
                                <div class="text-end mt-auto">
                                     <a href="vehicle_loans.php?vehicle_id=<?= $vehicle_id ?>" class="btn btn-sm btn-outline-info">Lihat Semua Peminjaman</a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div><!-- /.row -->

                <!-- Mileage History Card -->
                <div class="card mb-4">
                    <div class="card-header"><i class="fas fa-tachometer-alt"></i> Riwayat Kilometer</div>
                    <div class="card-body">
                        <?php if (!empty($mileage_history)): ?>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover table-sm">
                                    <thead>
                                        <tr>
                                            <th>Tanggal Catat</th>
                                            <th>Kilometer</th>
                                            <th>Catatan</th>
                                            <th>Ditambahkan</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($mileage_history as $entry): ?>
                                            <tr>
                                                <td><?= date('d M Y', strtotime($entry['reading_date'])) ?></td>
                                                <td><?= number_format($entry['mileage']) ?> KM</td>
                                                <td><?= nl2br(htmlspecialchars($entry['notes'] ?? '-')) ?></td>
                                                <td><?= date('d M Y H:i', strtotime($entry['created_at'])) ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <p class="text-center text-muted">Belum ada riwayat kilometer.</p>
                        <?php endif; ?>
                         <div class="text-end mt-2">
                             <a href="vehicle_mileage.php?vehicle_id=<?= $vehicle_id ?>" class="btn btn-sm btn-outline-info">Kelola Kilometer</a>
                        </div>
                    </div>
                </div>

                <!-- Maintenance & Tax History Card -->
                <div class="card mb-4">
                    <div class="card-header"><i class="fas fa-tools"></i> Riwayat Perawatan & Pajak</div>
                    <div class="card-body">
                         <?php if (!empty($event_history)): ?>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover table-sm">
                                    <thead>
                                        <tr>
                                            <th>Jenis</th>
                                            <th>Tanggal</th>
                                            <th>Deskripsi</th>
                                            <th>Biaya (Rp)</th>
                                            <th>Jatuh Tempo</th>
                                            <th>Catatan</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($event_history as $entry): ?>
                                            <?php
                                                $type_badge = $entry['event_type'] == 'maintenance' ? 'bg-warning text-dark' : 'bg-info text-dark';
                                                $is_due = $entry['next_due_date'] && $entry['next_due_date'] <= date('Y-m-d');
                                                $due_class = $is_due ? 'table-danger' : ''; 
                                            ?>
                                            <tr class="<?= $due_class ?>">
                                                <td><span class="badge <?= $type_badge ?>"><?= ucfirst($entry['event_type']) ?></span></td>
                                                <td><?= date('d M Y', strtotime($entry['event_date'])) ?></td>
                                                <td><?= htmlspecialchars($entry['description']) ?></td>
                                                <td class="text-end"><?= $entry['cost'] !== null ? number_format($entry['cost'], 2, ',', '.') : '-' ?></td>
                                                <td>
                                                    <?php if ($entry['next_due_date']): ?>
                                                        <?= date('d M Y', strtotime($entry['next_due_date'])) ?>
                                                        <?php if ($is_due): ?>
                                                            <span class="badge bg-danger ms-1">Due</span>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        -
                                                    <?php endif; ?>
                                                </td>
                                                <td><?= nl2br(htmlspecialchars($entry['notes'] ?? '-')) ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <p class="text-center text-muted">Belum ada riwayat data perawatan atau pajak.</p>
                        <?php endif; ?>
                         <div class="text-end mt-2">
                             <a href="vehicle_maintenance_tax.php?vehicle_id=<?= $vehicle_id ?>" class="btn btn-sm btn-outline-info">Kelola Perawatan/Pajak</a>
                        </div>
                    </div>
                </div>

                <?php endif; // end if($vehicle_data) ?>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>