<?php
require_once 'includes/config.php';
// Uncomment the line below if you need authentication
// require_once 'includes/auth.php';

$page_title = "Tambah Peminjaman Kendaraan";
$error_message = '';
$success_message = '';

// Fetch available vehicles for the dropdown
$vehicles_available = [];
$vehicles_query = "SELECT id, name, license_plate FROM vehicles WHERE status = 'available' ORDER BY name ASC";
$vehicles_result = $conn->query($vehicles_query);
if ($vehicles_result) {
    while ($vehicle = $vehicles_result->fetch_assoc()) {
        $vehicles_available[] = $vehicle;
    }
} else {
    $error_message = "Gagal mengambil daftar kendaraan: " . $conn->error;
    error_log("Error fetching available vehicles: " . $conn->error);
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // --- Data Retrieval ---
    $borrower_name = trim($_POST['borrower_name'] ?? '');
    $vehicle_id = filter_input(INPUT_POST, 'vehicle_id', FILTER_VALIDATE_INT);
    $start_date_str = trim($_POST['start_date'] ?? '');
    $end_date_str = trim($_POST['end_date'] ?? '');
    $purpose = trim($_POST['purpose'] ?? '');

    // --- Basic Validation ---
    if (empty($borrower_name) || empty($vehicle_id) || empty($start_date_str) || empty($end_date_str)) {
        $error_message = "Nama Peminjam, Kendaraan, Tanggal Mulai, dan Tanggal Selesai wajib diisi.";
    } else {
        // --- Date Validation ---
        $start_date = date('Y-m-d', strtotime($start_date_str));
        $end_date = date('Y-m-d', strtotime($end_date_str));
        $today = date('Y-m-d');

        if ($start_date < $today) {
            $error_message = "Tanggal Mulai tidak boleh tanggal yang sudah berlalu.";
        } elseif ($end_date < $start_date) {
            $error_message = "Tanggal Selesai tidak boleh sebelum Tanggal Mulai.";
        } else {
            // --- Check Vehicle Availability (Conflict Check) ---
            $conflict_sql = "
                SELECT id FROM vehicle_loans 
                WHERE vehicle_id = ? 
                  AND status IN ('scheduled', 'ongoing')
                  AND (
                      (start_date <= ? AND end_date >= ?) OR -- Overlaps the start
                      (start_date <= ? AND end_date >= ?) OR -- Overlaps the end
                      (start_date >= ? AND end_date <= ?)    -- Is contained within
                  )
                LIMIT 1";
            
            $stmt_check = $conn->prepare($conflict_sql);
            if ($stmt_check) {
                $stmt_check->bind_param("issssss", $vehicle_id, $start_date, $start_date, $end_date, $end_date, $start_date, $end_date);
                $stmt_check->execute();
                $stmt_check->store_result();
                if ($stmt_check->num_rows > 0) {
                    // Find vehicle name for error message
                    $vehicle_name_for_error = "Kendaraan";
                    foreach ($vehicles_available as $v) {
                        if ($v['id'] == $vehicle_id) {
                            $vehicle_name_for_error = $v['name'] . ' (' . $v['license_plate'] . ')';
                            break;
                        }
                    }
                     $error_message = "$vehicle_name_for_error sudah dijadwalkan pada periode tanggal tersebut.";
                }
                $stmt_check->close();
            } else {
                 $error_message = "Gagal memeriksa ketersediaan kendaraan: " . $conn->error;
                 error_log("Error checking vehicle availability: " . $conn->error);
            }

            // --- Proceed if no errors ---
            if (empty($error_message)) {
                $insert_sql = "INSERT INTO vehicle_loans (vehicle_id, borrower_name, start_date, end_date, purpose, status) 
                               VALUES (?, ?, ?, ?, ?, 'scheduled')";
                $stmt_insert = $conn->prepare($insert_sql);

                if ($stmt_insert) {
                    $stmt_insert->bind_param("issss", 
                        $vehicle_id, 
                        $borrower_name, 
                        $start_date, 
                        $end_date, 
                        $purpose
                    );

                    if ($stmt_insert->execute()) {
                        $success_message = "Peminjaman kendaraan berhasil dijadwalkan.";
                        // Optional: Redirect after success
                        // header("Location: vehicle_loans.php?status=added");
                        // exit;
                        
                        // Clear form fields for next entry (if not redirecting)
                        $_POST = []; 
                        $borrower_name = $vehicle_id = $start_date_str = $end_date_str = $purpose = '';

                    } else {
                        $error_message = "Gagal menambahkan jadwal peminjaman: " . $stmt_insert->error;
                        error_log("Error adding vehicle loan: " . $stmt_insert->error); 
                    }
                    $stmt_insert->close();
                } else {
                    $error_message = "Gagal mempersiapkan statement: " . $conn->error;
                    error_log("Error preparing vehicle loan insert statement: " . $conn->error); 
                }
            }
        }
    }
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4"><?= $page_title ?></h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="vehicle_loans.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-arrow-left"></i> Kembali ke Daftar Peminjaman
                        </a>
                    </div>
                </div>

                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($error_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <div class="card">
                    <div class="card-body">
                        <form action="vehicle_loan_add.php" method="POST">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="borrower_name" class="form-label">Nama Peminjam <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="borrower_name" name="borrower_name" value="<?= htmlspecialchars($_POST['borrower_name'] ?? '') ?>" required>
                                </div>
                                <div class="col-md-6">
                                    <label for="vehicle_id" class="form-label">Kendaraan <span class="text-danger">*</span></label>
                                    <select class="form-select" id="vehicle_id" name="vehicle_id" required>
                                        <option value="">-- Pilih Kendaraan --</option>
                                        <?php foreach ($vehicles_available as $vehicle): ?>
                                            <option value="<?= $vehicle['id'] ?>" <?= (($_POST['vehicle_id'] ?? '') == $vehicle['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($vehicle['name']) ?> (<?= htmlspecialchars($vehicle['license_plate']) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                        <?php if (empty($vehicles_available) && empty($error_message)): ?>
                                            <option value="" disabled>Tidak ada kendaraan yang tersedia</option>
                                        <?php endif; ?>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="start_date" class="form-label">Tanggal Mulai <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" id="start_date" name="start_date" value="<?= htmlspecialchars($_POST['start_date'] ?? '') ?>" required min="<?= date('Y-m-d') ?>">
                                </div>
                                <div class="col-md-6">
                                    <label for="end_date" class="form-label">Tanggal Selesai <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" id="end_date" name="end_date" value="<?= htmlspecialchars($_POST['end_date'] ?? '') ?>" required min="<?= date('Y-m-d') ?>">
                                </div>
                                <div class="col-12">
                                    <label for="purpose" class="form-label">Keperluan</label>
                                    <textarea class="form-control" id="purpose" name="purpose" rows="3"><?= htmlspecialchars($_POST['purpose'] ?? '') ?></textarea>
                                </div>
                            </div>

                            <hr class="my-4">

                            <button class="btn btn-primary" type="submit" <?= (empty($vehicles_available) ? 'disabled' : '') ?>> 
                                <i class="fas fa-calendar-plus"></i> Jadwalkan Peminjaman
                            </button>
                            <a href="vehicle_loans.php" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Batal
                            </a>
                        </form>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Basic client-side validation to ensure end date is not before start date
        const startDateInput = document.getElementById('start_date');
        const endDateInput = document.getElementById('end_date');

        if (startDateInput && endDateInput) {
            startDateInput.addEventListener('change', function() {
                if (this.value) {
                    endDateInput.min = this.value; // Set min attribute of end date
                    // If end date is already set and is before new start date, clear it (optional)
                    if (endDateInput.value && endDateInput.value < this.value) {
                        endDateInput.value = ''; 
                    }
                } else {
                     endDateInput.min = '<?= date('Y-m-d') ?>'; // Reset min if start date cleared
                }
            });
             // Set initial min value for end date if start date is pre-filled
            if (startDateInput.value) {
                 endDateInput.min = startDateInput.value;
            }
        }
    </script>
</body>
</html>