<?php
require_once 'includes/config.php';
// Uncomment the line below if you need authentication
// require_once 'includes/auth.php';

$page_title = "Data Perawatan & Pajak Kendaraan";
$error_message = '';
$success_message = '';

// Fetch all active vehicles for the dropdown
$vehicles_list = [];
$vehicles_query = "SELECT id, name, license_plate FROM vehicles WHERE status != 'inactive' ORDER BY name ASC";
$vehicles_result = $conn->query($vehicles_query);
if ($vehicles_result) {
    while ($vehicle = $vehicles_result->fetch_assoc()) {
        $vehicles_list[] = $vehicle;
    }
} else {
    error_log("Error fetching vehicles list: " . $conn->error);
}

// Handle form submission for adding events
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_event'])) {
    // --- Data Retrieval ---
    $vehicle_id = filter_input(INPUT_POST, 'vehicle_id', FILTER_VALIDATE_INT);
    $event_type = trim($_POST['event_type'] ?? '');
    $event_date_str = trim($_POST['event_date'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $cost_str = trim($_POST['cost'] ?? '');
    $next_due_date_str = trim($_POST['next_due_date'] ?? '');
    $notes = trim($_POST['notes'] ?? '');

    // --- Basic Validation ---
    if (empty($vehicle_id) || empty($event_type) || empty($event_date_str) || empty($description)) {
        $error_message = "Kendaraan, Jenis Event, Tanggal Event, dan Deskripsi wajib diisi.";
    } elseif (!in_array($event_type, ['maintenance', 'tax'])) {
         $error_message = "Jenis Event tidak valid.";
    } elseif ($cost_str !== '' && (!is_numeric($cost_str) || (float)$cost_str < 0)) {
        $error_message = "Biaya harus berupa angka positif atau kosong.";
    } else {
        $event_date = date('Y-m-d', strtotime($event_date_str));
        $cost = ($cost_str !== '') ? (float)$cost_str : null;
        $next_due_date = !empty($next_due_date_str) ? date('Y-m-d', strtotime($next_due_date_str)) : null;

        // Optional: Validate next_due_date is after event_date if provided
        if ($next_due_date !== null && $next_due_date < $event_date) {
             $error_message = "Tanggal Jatuh Tempo Berikutnya tidak boleh sebelum Tanggal Event.";
        }

        // --- Proceed if no errors ---
        if (empty($error_message)) {
            $insert_sql = "INSERT INTO vehicle_events (vehicle_id, event_type, event_date, description, cost, next_due_date, notes) 
                           VALUES (?, ?, ?, ?, ?, ?, ?)";
            $stmt_insert = $conn->prepare($insert_sql);

            if ($stmt_insert) {
                // Bind parameters: i=integer, s=string, d=double/decimal
                $stmt_insert->bind_param("isssdss", 
                    $vehicle_id, 
                    $event_type, 
                    $event_date, 
                    $description, 
                    $cost, // Pass null if empty
                    $next_due_date, // Pass null if empty
                    $notes
                );

                if ($stmt_insert->execute()) {
                    $success_message = "Data " . ucfirst($event_type) . " berhasil ditambahkan.";
                    // Clear potentially submitted values to reset form
                    $_POST = []; 
                } else {
                    $error_message = "Gagal menambahkan data event: " . $stmt_insert->error;
                    error_log("Error adding vehicle event: " . $stmt_insert->error); 
                }
                $stmt_insert->close();
            } else {
                $error_message = "Gagal mempersiapkan statement: " . $conn->error;
                error_log("Error preparing vehicle event insert statement: " . $conn->error); 
            }
        }
    }
}

// Fetch existing event data for display
// Consider adding filters (by vehicle, type, date range) and pagination later
$event_history = [];
$history_query = "
    SELECT 
        ve.id, ve.event_type, ve.event_date, ve.description, ve.cost, ve.next_due_date, ve.notes, ve.created_at,
        v.name as vehicle_name, v.license_plate
    FROM vehicle_events ve
    JOIN vehicles v ON ve.vehicle_id = v.id
    ORDER BY v.name ASC, ve.event_date DESC, ve.created_at DESC
";
$history_result = $conn->query($history_query);
if ($history_result) {
    while ($row = $history_result->fetch_assoc()) {
        $event_history[] = $row;
    }
} else {
    $error_message .= ($error_message ? '<br>' : '') . "Gagal mengambil riwayat event: " . $conn->error; // Append error
    error_log("Error fetching event history: " . $conn->error);
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4"><?= $page_title ?></h1>
                    <!-- Add buttons if needed, e.g., Export -->
                </div>

                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= $error_message ?> <!-- No htmlspecialchars needed if errors are controlled -->
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <!-- Add Event Form -->
                <div class="card mb-4">
                    <div class="card-header">
                        Input Data Perawatan / Pajak
                    </div>
                    <div class="card-body">
                        <form action="vehicle_maintenance_tax.php" method="POST">
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <label for="vehicle_id" class="form-label">Kendaraan <span class="text-danger">*</span></label>
                                    <select class="form-select" id="vehicle_id" name="vehicle_id" required>
                                        <option value="">-- Pilih Kendaraan --</option>
                                        <?php foreach ($vehicles_list as $vehicle): ?>
                                            <option value="<?= $vehicle['id'] ?>" <?= (($_POST['vehicle_id'] ?? '') == $vehicle['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($vehicle['name']) ?> (<?= htmlspecialchars($vehicle['license_plate']) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-6">
                                    <label for="event_type" class="form-label">Jenis Event <span class="text-danger">*</span></label>
                                    <select class="form-select" id="event_type" name="event_type" required>
                                        <option value="">-- Pilih Jenis --</option>
                                        <option value="maintenance" <?= (($_POST['event_type'] ?? '') == 'maintenance') ? 'selected' : '' ?>>Perawatan (Maintenance)</option>
                                        <option value="tax" <?= (($_POST['event_type'] ?? '') == 'tax') ? 'selected' : '' ?>>Pajak (Tax)</option>
                                    </select>
                                </div>
                                <div class="col-md-4">
                                    <label for="event_date" class="form-label">Tanggal Event <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" id="event_date" name="event_date" value="<?= htmlspecialchars($_POST['event_date'] ?? date('Y-m-d')) ?>" required>
                                    <div class="form-text">Tanggal pelaksanaan perawatan atau pembayaran pajak.</div>
                                </div>
                                 <div class="col-md-8">
                                    <label for="description" class="form-label">Deskripsi <span class="text-danger">*</span></label>
                                    <input type="text" class="form-control" id="description" name="description" value="<?= htmlspecialchars($_POST['description'] ?? '') ?>" required placeholder="Contoh: Ganti Oli, Servis Rutin, Pajak Tahunan STNK">
                                </div>
                                <div class="col-md-4">
                                    <label for="cost" class="form-label">Biaya (Rp)</label>
                                    <input type="number" class="form-control" id="cost" name="cost" min="0" step="0.01" value="<?= htmlspecialchars($_POST['cost'] ?? '') ?>" placeholder="Kosongkan jika tidak ada">
                                </div>
                                <div class="col-md-4">
                                    <label for="next_due_date" class="form-label">Jatuh Tempo Berikutnya</label>
                                    <input type="date" class="form-control" id="next_due_date" name="next_due_date" value="<?= htmlspecialchars($_POST['next_due_date'] ?? '') ?>">
                                    <div class="form-text">Untuk reminder perawatan/pajak selanjutnya.</div>
                                </div>
                                <div class="col-md-4">
                                    <label for="notes" class="form-label">Catatan Tambahan</label>
                                    <input type="text" class="form-control" id="notes" name="notes" value="<?= htmlspecialchars($_POST['notes'] ?? '') ?>">
                                </div>
                                <div class="col-12 mt-3">
                                    <button class="btn btn-primary" type="submit" name="add_event">
                                        <i class="fas fa-save"></i> Simpan Data
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Event History Table -->
                <div class="card">
                    <div class="card-header">
                        Riwayat Perawatan & Pajak
                    </div>
                    <div class="card-body">
                        <?php if (!empty($event_history)): ?>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover table-sm">
                                    <thead>
                                        <tr>
                                            <th>Kendaraan</th>
                                            <th>Jenis</th>
                                            <th>Tanggal</th>
                                            <th>Deskripsi</th>
                                            <th>Biaya (Rp)</th>
                                            <th>Jatuh Tempo</th>
                                            <th>Catatan</th>
                                            <th>Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($event_history as $entry): ?>
                                            <?php
                                                $type_badge = $entry['event_type'] == 'maintenance' ? 'bg-warning text-dark' : 'bg-info text-dark';
                                                $is_due = $entry['next_due_date'] && $entry['next_due_date'] <= date('Y-m-d');
                                                $due_class = $is_due ? 'table-danger' : ''; // Highlight overdue rows
                                            ?>
                                            <tr class="<?= $due_class ?>">
                                                <td><?= htmlspecialchars($entry['vehicle_name']) ?> (<?= htmlspecialchars($entry['license_plate']) ?>)</td>
                                                <td><span class="badge <?= $type_badge ?>"><?= ucfirst($entry['event_type']) ?></span></td>
                                                <td><?= date('d M Y', strtotime($entry['event_date'])) ?></td>
                                                <td><?= htmlspecialchars($entry['description']) ?></td>
                                                <td class="text-end"><?= $entry['cost'] !== null ? number_format($entry['cost'], 2, ',', '.') : '-' ?></td>
                                                <td>
                                                    <?php if ($entry['next_due_date']): ?>
                                                        <?= date('d M Y', strtotime($entry['next_due_date'])) ?>
                                                        <?php if ($is_due): ?>
                                                            <span class="badge bg-danger ms-1">Due</span>
                                                        <?php endif; ?>
                                                    <?php else: ?>
                                                        -
                                                    <?php endif; ?>
                                                </td>
                                                <td><?= nl2br(htmlspecialchars($entry['notes'] ?? '-')) ?></td>
                                                <td>
                                                    <!-- Add Edit/Delete buttons later if needed -->
                                                    <a href="vehicle_event_edit.php?id=<?= $entry['id'] ?>" class="btn btn-xs btn-outline-primary disabled" title="Edit (Belum Tersedia)">
                                                        <i class="fas fa-pencil-alt"></i>
                                                    </a>
                                                    <a href="vehicle_event_delete.php?id=<?= $entry['id'] ?>" class="btn btn-xs btn-outline-danger ms-1 disabled" title="Hapus (Belum Tersedia)" onclick="return confirm('Yakin ingin menghapus data event ini?');">
                                                        <i class="fas fa-trash-alt"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Add pagination controls here if needed -->
                        <?php else: ?>
                            <p class="text-center text-muted">Belum ada riwayat data perawatan atau pajak.</p>
                        <?php endif; ?>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>