<?php
require_once 'includes/config.php';
// Uncomment the line below if you need authentication
// require_once 'includes/auth.php';

$page_title = "Data Kilometer Kendaraan";
$error_message = '';
$success_message = '';

// Fetch all active vehicles for the dropdown
$vehicles_list = [];
$vehicles_query = "SELECT id, name, license_plate FROM vehicles WHERE status != 'inactive' ORDER BY name ASC";
$vehicles_result = $conn->query($vehicles_query);
if ($vehicles_result) {
    while ($vehicle = $vehicles_result->fetch_assoc()) {
        $vehicles_list[] = $vehicle;
    }
} else {
    // Non-fatal error, form might still work but dropdown will be empty
    error_log("Error fetching vehicles list: " . $conn->error);
}

// Handle form submission for adding mileage
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_mileage'])) {
    // --- Data Retrieval ---
    $vehicle_id = filter_input(INPUT_POST, 'vehicle_id', FILTER_VALIDATE_INT);
    $reading_date_str = trim($_POST['reading_date'] ?? '');
    $mileage_str = trim($_POST['mileage'] ?? '');
    $notes = trim($_POST['notes'] ?? '');

    // --- Basic Validation ---
    if (empty($vehicle_id) || empty($reading_date_str) || $mileage_str === '') {
        $error_message = "Kendaraan, Tanggal Pencatatan, dan Angka Kilometer wajib diisi.";
    } elseif (!is_numeric($mileage_str) || (int)$mileage_str < 0) {
        $error_message = "Angka Kilometer harus berupa angka positif.";
    } else {
        $reading_date = date('Y-m-d', strtotime($reading_date_str));
        $mileage = (int)$mileage_str;

        // --- Check for duplicate entry (vehicle_id, reading_date) ---
        // Although DB has UNIQUE constraint, checking here provides a friendlier error
        $check_sql = "SELECT id FROM vehicle_mileage WHERE vehicle_id = ? AND reading_date = ?";
        $stmt_check = $conn->prepare($check_sql);
        if ($stmt_check) {
            $stmt_check->bind_param("is", $vehicle_id, $reading_date);
            $stmt_check->execute();
            $stmt_check->store_result();
            if ($stmt_check->num_rows > 0) {
                $error_message = "Data kilometer untuk kendaraan ini pada tanggal tersebut sudah ada.";
            }
            $stmt_check->close();
        } else {
            $error_message = "Gagal memeriksa duplikasi data: " . $conn->error;
            error_log("Error checking duplicate mileage: " . $conn->error);
        }

        // --- Proceed if no errors ---
        if (empty($error_message)) {
            $insert_sql = "INSERT INTO vehicle_mileage (vehicle_id, reading_date, mileage, notes) 
                           VALUES (?, ?, ?, ?)";
            $stmt_insert = $conn->prepare($insert_sql);

            if ($stmt_insert) {
                $stmt_insert->bind_param("isis", 
                    $vehicle_id, 
                    $reading_date, 
                    $mileage, 
                    $notes
                );

                if ($stmt_insert->execute()) {
                    $success_message = "Data kilometer berhasil ditambahkan.";
                    // Clear potentially submitted values to reset form
                    $_POST = []; 
                } else {
                    // Check for specific duplicate key error from DB constraint
                    if ($conn->errno == 1062) { // Error code for duplicate entry
                         $error_message = "Data kilometer untuk kendaraan ini pada tanggal tersebut sudah ada (DB constraint).";
                    } else {
                        $error_message = "Gagal menambahkan data kilometer: " . $stmt_insert->error;
                    }
                    error_log("Error adding vehicle mileage: " . $stmt_insert->error); 
                }
                $stmt_insert->close();
            } else {
                $error_message = "Gagal mempersiapkan statement: " . $conn->error;
                error_log("Error preparing vehicle mileage insert statement: " . $conn->error); 
            }
        }
    }
}

// Fetch existing mileage data for display
// Consider adding filters (by vehicle, date range) and pagination later
$mileage_history = [];
$history_query = "
    SELECT 
        vm.id, vm.reading_date, vm.mileage, vm.notes, vm.created_at,
        v.name as vehicle_name, v.license_plate
    FROM vehicle_mileage vm
    JOIN vehicles v ON vm.vehicle_id = v.id
    ORDER BY v.name ASC, vm.reading_date DESC
";
$history_result = $conn->query($history_query);
if ($history_result) {
    while ($row = $history_result->fetch_assoc()) {
        $mileage_history[] = $row;
    }
} else {
    $error_message .= ($error_message ? '<br>' : '') . "Gagal mengambil riwayat kilometer: " . $conn->error; // Append error
    error_log("Error fetching mileage history: " . $conn->error);
}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $page_title ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4"><?= $page_title ?></h1>
                    <!-- Add buttons if needed, e.g., Export -->
                </div>

                <?php if ($error_message): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?= $error_message ?> <!-- No htmlspecialchars needed if errors are controlled -->
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                <?php if ($success_message): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?= htmlspecialchars($success_message) ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>

                <!-- Add Mileage Form -->
                <div class="card mb-4">
                    <div class="card-header">
                        Input Data Kilometer Bulanan
                    </div>
                    <div class="card-body">
                        <form action="vehicle_mileage.php" method="POST">
                            <div class="row g-3 align-items-end">
                                <div class="col-md-4">
                                    <label for="vehicle_id" class="form-label">Kendaraan <span class="text-danger">*</span></label>
                                    <select class="form-select" id="vehicle_id" name="vehicle_id" required>
                                        <option value="">-- Pilih Kendaraan --</option>
                                        <?php foreach ($vehicles_list as $vehicle): ?>
                                            <option value="<?= $vehicle['id'] ?>" <?= (($_POST['vehicle_id'] ?? '') == $vehicle['id']) ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($vehicle['name']) ?> (<?= htmlspecialchars($vehicle['license_plate']) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label for="reading_date" class="form-label">Tanggal Pencatatan <span class="text-danger">*</span></label>
                                    <input type="date" class="form-control" id="reading_date" name="reading_date" value="<?= htmlspecialchars($_POST['reading_date'] ?? date('Y-m-d')) ?>" required>
                                </div>
                                <div class="col-md-2">
                                    <label for="mileage" class="form-label">KM Terakhir <span class="text-danger">*</span></label>
                                    <input type="number" class="form-control" id="mileage" name="mileage" min="0" step="1" value="<?= htmlspecialchars($_POST['mileage'] ?? '') ?>" required>
                                </div>
                                <div class="col-md-3">
                                    <label for="notes" class="form-label">Catatan</label>
                                    <input type="text" class="form-control" id="notes" name="notes" value="<?= htmlspecialchars($_POST['notes'] ?? '') ?>">
                                </div>
                                <div class="col-12 mt-3">
                                    <button class="btn btn-primary" type="submit" name="add_mileage">
                                        <i class="fas fa-save"></i> Simpan Data KM
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Mileage History Table -->
                <div class="card">
                    <div class="card-header">
                        Riwayat Data Kilometer
                    </div>
                    <div class="card-body">
                        <?php if (!empty($mileage_history)): ?>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover table-sm">
                                    <thead>
                                        <tr>
                                            <th>Kendaraan</th>
                                            <th>Tanggal Catat</th>
                                            <th>Kilometer</th>
                                            <th>Catatan</th>
                                            <th>Ditambahkan Pada</th>
                                            <th>Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($mileage_history as $entry): ?>
                                            <tr>
                                                <td><?= htmlspecialchars($entry['vehicle_name']) ?> (<?= htmlspecialchars($entry['license_plate']) ?>)</td>
                                                <td><?= date('d M Y', strtotime($entry['reading_date'])) ?></td>
                                                <td><?= number_format($entry['mileage']) ?> KM</td>
                                                <td><?= nl2br(htmlspecialchars($entry['notes'] ?? '-')) ?></td>
                                                <td><?= date('d M Y H:i', strtotime($entry['created_at'])) ?></td>
                                                <td>
                                                    <!-- Add Edit/Delete buttons later if needed -->
                                                    <a href="vehicle_mileage_edit.php?id=<?= $entry['id'] ?>" class="btn btn-xs btn-outline-primary disabled" title="Edit (Belum Tersedia)">
                                                        <i class="fas fa-pencil-alt"></i>
                                                    </a>
                                                    <a href="vehicle_mileage_delete.php?id=<?= $entry['id'] ?>" class="btn btn-xs btn-outline-danger ms-1 disabled" title="Hapus (Belum Tersedia)" onclick="return confirm('Yakin ingin menghapus data kilometer ini?');">
                                                        <i class="fas fa-trash-alt"></i>
                                                    </a>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Add pagination controls here if needed -->
                        <?php else: ?>
                            <p class="text-center text-muted">Belum ada riwayat data kilometer.</p>
                        <?php endif; ?>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>