<?php
require_once 'includes/config.php';
if (!$dev_mode) {
    require_once 'includes/auth.php';
}
$page_title = "Dashboard Jadwal Kendaraan";
$current_page = 'vehicle_schedule'; // For sidebar active state

// --- Fetch Data for Calendar ---
$calendar_events = [];
$today = date('Y-m-d');
$one_month_ago = date('Y-m-d', strtotime('-1 month'));

// 1. Fetch Vehicle Loans (Scheduled and Ongoing)
$loans_query = "
    SELECT
        vl.id, vl.borrower_name, vl.start_date, vl.end_date, vl.purpose, vl.status,
        v.name as vehicle_name, v.license_plate, v.id as vehicle_id
    FROM vehicle_loans vl
    JOIN vehicles v ON vl.vehicle_id = v.id
    WHERE vl.status IN ('scheduled', 'ongoing')
      AND vl.end_date >= ?
    ORDER BY vl.start_date ASC
";
$stmt_loans = $conn->prepare($loans_query);
if ($stmt_loans) {
    $stmt_loans->bind_param("s", $one_month_ago); // Show loans ending from one month ago onwards
    $stmt_loans->execute();
    $loans_result = $stmt_loans->get_result();
    while ($loan = $loans_result->fetch_assoc()) {
        $calendar_events[] = [
            'id' => 'loan_' . $loan['id'],
            'title' => "Pinjam: " . $loan['vehicle_name'] . " (" . $loan['borrower_name'] . ")",
            'start' => $loan['start_date'],
            'end' => date('Y-m-d', strtotime($loan['end_date'] . ' +1 day')), // FullCalendar end date is exclusive
            'color' => ($loan['status'] == 'ongoing') ? '#28a745' : '#ffc107', // green (ongoing), yellow (scheduled)
            'url' => 'vehicle_loan_detail.php?id=' . $loan['id'], // Link to loan detail
            'extendedProps' => [
                'type' => 'loan',
                'status' => $loan['status'],
                'borrower' => $loan['borrower_name'],
                'vehicle' => $loan['vehicle_name'] . ' (' . $loan['license_plate'] . ')',
                'purpose' => $loan['purpose']
            ]
        ];
    }
    $stmt_loans->close();
} else {
    error_log("Error preparing loans query for calendar: " . $conn->error);
}

# Update the table name from vehicle_maintenance to vehicle_events }}
// 2. Fetch Maintenance/Tax Due Dates (Upcoming or Recently Due) from vehicle_events table
$events_query = "
    SELECT
        ve.id, ve.vehicle_id, ve.event_type, ve.description, ve.next_due_date,
        v.name as vehicle_name, v.license_plate
    FROM vehicle_events ve  -- Changed table name here
    JOIN vehicles v ON ve.vehicle_id = v.id
    WHERE ve.next_due_date IS NOT NULL AND ve.next_due_date >= ?
    ORDER BY ve.next_due_date ASC
";
$stmt_events = $conn->prepare($events_query); // This line might have caused an error before if the table name was wrong
if ($stmt_events) {
    $stmt_events->bind_param("s", $one_month_ago); // Show events due from one month ago onwards
    $stmt_events->execute();
    $events_result = $stmt_events->get_result();
    while ($event = $events_result->fetch_assoc()) {
        $is_due = $event['next_due_date'] <= $today;
        $vehicle_display = $event['vehicle_name'] . ' (' . $event['license_plate'] . ')';
        // {{ This is where vehicle events are added to the $calendar_events array }}
        $calendar_events[] = [
            'id' => 'event_due_' . $event['id'],
            'title' => ucfirst($event['event_type']) . " Due: " . $event['vehicle_name'] . " - " . ($event['description'] ?? 'N/A'), // Added null check for description
            'start' => $event['next_due_date'], // The date the event appears on
            'color' => $is_due ? '#dc3545' : (($event['event_type'] == 'maintenance') ? '#fd7e14' : '#20c997'), // red (due), orange (maint due), cyan (tax due)
            'url' => 'vehicle_detail.php?id=' . $event['vehicle_id'], // Link to vehicle detail page
            'extendedProps' => [
                'type' => $event['event_type'] . '_due',
                'vehicle' => $vehicle_display,
                'description' => $event['description'] ?? '' // Ensure description exists
            ]
        ];
    }
    $stmt_events->close();
} else {
    // Log the specific error if preparation fails
    error_log("Error preparing vehicle_events query for calendar: " . $conn->error);
}


// --- Fetch Data for Table View ---
$loans_table_query = "
    SELECT
        vl.id, vl.borrower_name, vl.start_date, vl.end_date, vl.purpose, vl.status,
        v.name as vehicle_name, v.license_plate
    FROM vehicle_loans vl
    JOIN vehicles v ON vl.vehicle_id = v.id
    WHERE vl.end_date >= ?
      AND vl.status IN ('scheduled', 'ongoing')
    ORDER BY vl.start_date ASC, v.name ASC
";
$stmt_table = $conn->prepare($loans_table_query);
$loans_table_result = null; // Initialize
if ($stmt_table) {
    $stmt_table->bind_param("s", $today); // Show loans ending today or later for the table
    $stmt_table->execute();
    $loans_table_result = $stmt_table->get_result();
    // Don't close yet, needed for the loop below
} else {
    error_log("Error preparing loans table query: " . $conn->error);
}
# Add code to fetch all vehicle events for debugging #}}
// --- Fetch ALL Vehicle Events for Debug Display (if dev_mode is on) ---
$all_vehicle_events = [];
if ($dev_mode) {
    $all_events_query = "SELECT * FROM vehicle_events ORDER BY vehicle_id, event_type, next_due_date DESC";
    $all_events_result = $conn->query($all_events_query);
    if ($all_events_result) {
        while ($row = $all_events_result->fetch_assoc()) {
            $all_vehicle_events[] = $row;
        }
        $all_events_result->free(); // Free result set
    } else {
        error_log("Error fetching all vehicle_events for debug: " . $conn->error);
    }
}
# End of new code #}}

?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($page_title) ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="../assets/css/style.css" rel="stylesheet">
    <link href="../assets/css/dashboard.css" rel="stylesheet">
    <link href='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/main.min.css' rel='stylesheet' />
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        #vehicle-calendar {
            max-width: 1100px;
            margin: 20px auto;
        }
        .fc-event {
            cursor: pointer; /* Make events clickable */
        }
        /* Style for list view items */
        .fc-list-event {
            padding-top: 5px !important;
            padding-bottom: 5px !important;
        }
        .fc-list-event-title a {
            color: inherit; /* Make link color same as event text */
            text-decoration: none;
        }
        .fc-list-event-title a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <?php include 'includes/header.php'; ?>

    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>

            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4 py-3">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pb-2 mb-3 border-bottom">
                    <h1 class="h4"><?= htmlspecialchars($page_title) ?></h1>
                    <div class="btn-toolbar mb-2 mb-md-0">
                        <a href="vehicle_loans.php" class="btn btn-sm btn-outline-secondary">
                            <i class="fas fa-list"></i> Lihat Semua Peminjaman
                        </a>
                         <a href="vehicle_loan_add.php" class="btn btn-sm btn-primary ms-2">
                            <i class="fas fa-plus"></i> Tambah Peminjaman
                        </a>
                    </div>
                </div>

                <!-- Calendar View -->
                <div class="card mb-4">
                    <div class="card-header">
                        Kalender Jadwal Kendaraan
                    </div>
                    <div class="card-body">
                        <div id="vehicle-calendar"></div>
                    </div>
                </div>

                
                <?php if ($dev_mode && !empty($all_vehicle_events)): ?>
                <div class="card mb-4">
                    <div class="card-header bg-info text-dark">
                        DEBUG: Raw Data from `vehicle_events` Table
                    </div>
                    <div class="card-body">
                        
                        <div class="table-responsive">
                            <table class="table table-sm table-bordered table-striped">
                                <thead class="table-light">
                                    <tr>
                                        <th>ID</th>
                                        <th>Vehicle ID</th>
                                        <th>Type</th>
                                        <th>Event Date</th>
                                        <th>Description</th>
                                        <th>Cost</th>
                                        <th>Next Due</th>
                                        <th>Notes</th>
                                        <th>Created At</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($all_vehicle_events as $debug_event): ?>
                                        <tr>
                                            <td><?= htmlspecialchars($debug_event['id']) ?></td>
                                            <td><?= htmlspecialchars($debug_event['vehicle_id']) ?></td>
                                            <td><?= htmlspecialchars(ucfirst($debug_event['event_type'])) ?></td>
                                            <td><?= htmlspecialchars($debug_event['event_date'] ? date('d M Y', strtotime($debug_event['event_date'])) : '-') ?></td>
                                            <td><?= nl2br(htmlspecialchars($debug_event['description'] ?? '-')) ?></td>
                                            <td><?= htmlspecialchars($debug_event['cost'] ? number_format($debug_event['cost'], 2, ',', '.') : '-') ?></td>
                                            <td><?= htmlspecialchars($debug_event['next_due_date'] ? date('d M Y', strtotime($debug_event['next_due_date'])) : '-') ?></td>
                                            <td><?= nl2br(htmlspecialchars($debug_event['notes'] ?? '-')) ?></td>
                                            <td><?= htmlspecialchars($debug_event['created_at'] ? date('d M Y H:i', strtotime($debug_event['created_at'])) : '-') ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                    </div>
                </div>
                <?php elseif ($dev_mode): ?>
                <div class="alert alert-info">No data found in `vehicle_events` or error fetching.</div>
                <?php endif; ?>
                


                <!-- Schedule Table Area -->
                <div class="card">
                    <div class="card-header">
                        Jadwal Aktif & Akan Datang (Tabel)
                    </div>
                    <div class="card-body">
                        <?php if ($loans_table_result && $loans_table_result->num_rows > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-striped table-hover table-sm">
                                    <thead>
                                        <tr>
                                            <th>Kendaraan</th>
                                            <th>Peminjam</th>
                                            <th>Tanggal Mulai</th>
                                            <th>Tanggal Selesai</th>
                                            <th>Keperluan</th>
                                            <th>Status</th>
                                            <th>Aksi</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while($loan = $loans_table_result->fetch_assoc()): ?>
                                            <?php
                                                $status_badge = 'bg-secondary'; // Default
                                                if ($loan['status'] == 'scheduled') { // Line 112 (approx)
                                                    $status_badge = 'bg-warning text-dark';
                                                } elseif ($loan['status'] == 'ongoing') {
                                                    $status_badge = 'bg-success';
                                                } // {{ Remove any comments on this line and ensure it only contains the closing brace }}
                                            ?>
                                            <tr>
                                                <td><?= htmlspecialchars($loan['vehicle_name']) ?> (<?= htmlspecialchars($loan['license_plate']) ?>)</td>
                                                <td><?= htmlspecialchars($loan['borrower_name']) ?></td>
                                                <td><?= date('d M Y', strtotime($loan['start_date'])) ?></td>
                                                <td><?= date('d M Y', strtotime($loan['end_date'])) ?></td>
                                                <td><?= nl2br(htmlspecialchars($loan['purpose'] ?? '-')) ?></td>
                                                <td><span class="badge <?= $status_badge ?>"><?= ucfirst($loan['status']) ?></span></td>
                                                <td>
                                                    <a href="vehicle_loan_detail.php?id=<?= $loan['id'] ?>" class="btn btn-xs btn-outline-info" title="Lihat Detail">
                                                        <i class="fas fa-eye"></i>
                                                    </a>
                                                    <!-- Add Edit/Cancel buttons if needed -->
                                                </td>
                                            </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <p class="text-center text-muted">Tidak ada jadwal peminjaman aktif atau yang akan datang untuk ditampilkan di tabel.</p>
                        <?php endif; ?>
                        <?php
                            // Close the statement for the table result now
                            if ($stmt_table) {
                                $stmt_table->close();
                            } elseif (!$loans_table_result) { // Check if result itself is null (query prep failed)
                                echo '<p class="text-center text-danger">Terjadi kesalahan saat mengambil data peminjaman untuk tabel.</p>';
                            }
                        ?>
                    </div>
                </div>

            </main>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src='https://cdn.jsdelivr.net/npm/fullcalendar@6.1.10/index.global.min.js'></script>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        const calendarEl = document.getElementById('vehicle-calendar');

        if (!calendarEl) {
            console.error("Calendar element #vehicle-calendar not found!");
            return;
        }
         if (typeof FullCalendar === 'undefined') {
            console.error("FullCalendar library is not loaded!");
             calendarEl.innerHTML = '<div class="alert alert-danger">Error: Calendar library failed to load.</div>';
            return;
        }

        const calendar = new FullCalendar.Calendar(calendarEl, {
            // Appearance settings similar to index.php
            initialView: 'dayGridMonth',
            headerToolbar: false, // No prev/next/title/view buttons
            height: 'auto',
            fixedWeekCount: false,
            dayMaxEvents: 2, // Show "+X more" if more than 2 events
            // eventDisplay: 'list-item', // Current setting: lists event inside each day cell
            // Consider removing or changing eventDisplay if you want a visual block spanning days
            // Default ('auto') usually shows blocks for multi-day events in month view.

            // Event data source (embedded PHP)
            // {{ The $calendar_events array (containing both loans and vehicle events) is used here }}
            events: [
                <?php
                foreach ($calendar_events as $event):
                    if (empty($event['title']) || empty($event['start'])) { continue; }
                ?>
                {
                    id: '<?= addslashes($event['id'] ?? '') ?>',
                    title: '<?= addslashes($event['title']) ?>',
                    start: '<?= addslashes($event['start']) ?>',
                    <?php if (!empty($event['end'])): ?>
                    end: '<?= addslashes($event['end']) ?>',
                    <?php endif; ?>
                    <?php if (!empty($event['color'])): ?>
                    color: '<?= addslashes($event['color']) ?>',
                    <?php endif; ?>
                    <?php if (!empty($event['url'])): ?>
                    url: '<?= addslashes($event['url']) ?>',
                    <?php endif; ?>
                    extendedProps: <?= json_encode($event['extendedProps'] ?? new stdClass(), JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT | JSON_HEX_AMP | JSON_UNESCAPED_UNICODE) ?>
                },
                <?php endforeach; ?>
            ],

            // Handle event clicks
            eventClick: function(info) {
                info.jsEvent.preventDefault(); // Prevent browser navigation
                if (info.event.url) {
                    window.open(info.event.url, "_self"); // Navigate in the same tab
                } else {
                    console.log("No URL defined for clicked event:", info.event);
                }
            },

            // Optional: Loading indicator
            loading: function(isLoading) {
                calendarEl.style.opacity = isLoading ? 0.5 : 1; // Simple visual feedback
                console.log('Calendar loading:', isLoading);
            }
        });

        try {
            calendar.render();
            console.log("Calendar rendered successfully.");
        } catch (e) {
            console.error("Error rendering FullCalendar:", e);
            calendarEl.innerHTML = '<div class="alert alert-danger">Error rendering calendar. Check console.</div>';
        }
    });
</script>
</body>
</html>